(function () {
    function parseCertsXml(rawXml) {
        var x = KS.xml2json.parser(rawXml);
        if (!KS.isEmpty(x.error)) {
            KS.error(x.error);
            return null;
        }
        x = x.certificates.certificate;

        function fillRow(cert) {
            var row = "".split("");
            row[0] = cert.issuer;
            row[1] = $.trim(cert.notafter);
            row[2] = $.trim(cert.notbefore);
            row[3] = cert.serial;
            row[4] = cert.subject;
            row[5] = cert.message;
            row[6] = cert.error;
            row[7] = cert.subjectx500;

            if (KS.isEmpty(row[5])) row[5] = "";
            if (KS.isEmpty(row[6])) row[6] = "";

            return row;
        }

        var data = "".split("");

        if (x == null || x === "undefined") { // No certificates
        } else if (x.length == null || x.length === "undefined") { // Only one certificate
            data[0] = fillRow(x);
        } else {
            for (var i = 0; i < x.length; i++) {
                data[i] = fillRow(x[i]);
            }
        }

        return data;
    }

    function fillCertsGridStore(certGrid, certs, enableErrs) {
        var isEmpty = true;

        KS.XCrypt.certsList = [];
        if (KS.isEmpty(certs)) return true;

        for (var cidx = 0; cidx < certs.length; cidx++) {
            var cert = certs[cidx],
                values = {},
                value,
                isError = false,
                isFilterErroneousCerts = KS.XCrypt.isFilterErroneousCerts,
                isAddToCertsList = true;

            // CryptoModule and Cades returns cert as object
            for (var field in cert) {
                if (cert.hasOwnProperty(field)) {
                    value = cert[field];
                    switch (field) {
                        case 'error':
                        case 'filterreason':
                            if (!KS.isEmpty(value)) isError = true;
                            break;
                        case 'notafter':
                        case 'notbefore':
                            value = KS.formatDate(new Date(value));
                            break;
                        case 'issuer':
                            if (KS.isObject(cert['issuerx500'])) {
                                value = cert['issuerx500']['CN'];
                            }
                            break;
                        case 'subject':
                            if (KS.isObject(cert['subjectx500'])) {
                                var fio = cert['subjectx500']['FIO'];
                                value = cert['subjectx500']['CN'];
                                if (!KS.isEmpty(fio) && value !== fio)
                                    value += ' (' + fio + ')';
                            }
                            break;
                        case 'subjectx500':
                            if (KS.isObject(value)) {
                                var fio = getCertPart(value, "FIO"),
                                    cn = getCertPart(value, "CN"),
                                    sn = getCertPart(value, "SN"),
                                    g = getCertPart(value, "G"),
                                    inn = getCertPart(value, "INN");
                                if (!KS.isEmpty(fio)) {
                                    value = fio;
                                } else if (!KS.isEmpty(cn)) {
                                    value = cn;
                                } else {
                                    value = sn + ' ' + g;
                                }

                                if (!KS.isEmpty(fio)) {
                                    values["inn"] = inn;
                                }
                            }
                            break;
                    }
                    values[field] = value;
                }
            }
            
            if (isFilterErroneousCerts) {
                isAddToCertsList = !isError;
            }
            
            if (isAddToCertsList){
                KS.XCrypt.certsList.push(values);

                if (!KS.isEmpty(KS.XCrypt.signSettings.keyId) && !certIsRequired(values)) {
                } else {
                    certGrid.addRecord(values);
                    isEmpty = false;
                }
            }
        }

        if (enableErrs !== true) {
            certGrid.on('beforeselect', certGridBeforeSelectHandler);
        }
        certGrid.getView().getRowClass = certGridGetRowClass;

        return isEmpty;
    }

    function getCertPart(subjectx500, partName) {
        var val = subjectx500;
        if (KS.isEmpty(val)) return '';
        if (KS.isString(val)) {
            if (val.indexOf(partName + '=') >= 0) {
                val = val.substring(val.indexOf(partName + '=') + 3).split(',')[0];
            }
        } else if (val) {
            val = val[partName];
        }
        return val || '';
    }

    function certIsRequired(cert) {
        var keys = KS.XCrypt.signSettings.keyId;
        if (KS.isEmpty(keys)) return false;

        var keysArr = keys.split(';').map(x => x.trim().toLowerCase());
        var isRequired = false;
        for (var i = 0; i < keysArr.length; i++){
            if (KS.isEmpty(keysArr[i])) continue;
            if (keysArr[i][0] == "#") {
                var ser = cert['serial'];
                var normSer = ser.replace('#', '').replace('?', '').replace(/ /g, '').toUpperCase(),
                    normKey = keysArr[i].replace('#', '').replace('?', '').replace(/ /g, '').toUpperCase();
                isRequired = normSer === normKey;
            } else {
                var cn = getCertPart(cert['subjectx500'], "CN").toLowerCase(),
                    sn = getCertPart(cert['subjectx500'], "SN").toLowerCase(),
                    g = getCertPart(cert['subjectx500'], "G").toLowerCase();

                isRequired = cn.indexOf(keysArr[i]) >= 0 || sn.indexOf(keysArr[i]) >= 0 || (sn + ' ' + g).indexOf(keysArr[i]) >= 0;
            }
            if (isRequired) return true;
        }

        return isRequired;
    }

    function getCertError(record) {
        var error = KS.Grid.getAnyCase(record, 'error');
        if (KS.isEmpty(error) && KS.edsType === 2) {
            error = KS.Grid.getAnyCase(record, 'filterreason');
        }
        return error;
    }

    function certGridBeforeSelectHandler(selModel, record) {
        var error = getCertError(record);
        if (!KS.isEmpty(error)) {
            KS.msg(error);
            return false;
        }
    }

    KS.registerCssStyles([
        {
            Properties: { 'background-color': '#ffb6c1' },
            Selector: ".cert-grid-err-record"
        }
    ]);

    function certGridGetRowClass(record) {
        var error = getCertError(record);
        if (!KS.isEmpty(error))
            return 'cert-grid-err-record';
        return '';
    }

    function getSignerFilterString(keyId) {
        var ss = KS.XCrypt.signSettings,
            filter = "";

        if (!ss.noNeedFilter) {
            if (!KS.isEmpty(keyId)) {
                if (!KS.isEmpty(filter)) {
                    filter += "; ";
                }
                filter += KS.L10n.sign_ID + keyId;
            }
            if (!KS.isEmpty(ss.issuerFilter)) {
                if (!KS.isEmpty(filter)) filter += "; ";
                filter += KS.L10n.sign_Supplier + ss.issuerFilter;
            }
            if (!KS.isEmpty(ss.filterOid)) {
                if (!KS.isEmpty(filter)) filter += "; ";
                filter += KS.L10n.sign_Appointment + ss.filterOid;
            }
            if (!KS.isEmpty(ss.secretKeyValidity)) {
                if (!KS.isEmpty(filter)) filter += "; ";
                filter += KS.L10n.sign_ValidityPeriod + ss.secretKeyValidity;
            }
            if (!KS.isEmpty(filter)) filter = KS.L10n.sign_SystemFilter + filter;
        }
        return filter;
    }

    function prepareCertIdForDisplay(certId) {
        if (KS.isEmpty(certId)) return '';
        if (certId.length > 2 && certId.substring(0, 1) == '#') {
            var id = certId.replace('#', '').replace('?', '').replace(/ /g, ''),
                result = '';
            if (id.length % 2 == 0) {
                for (var idx = 0; idx < id.length; idx += 2) {
                    result += id[idx] + id[idx + 1] + ' ';
                }
                certId = result;
            }
        }
        return certId.replace('?', '');
    }

    function displayCertsList(certs, fn, scope, enableErrs) {
        var ss = KS.XCrypt.signSettings;

        KS.XCrypt.certGrid = KS.create(ss.сertGrid);

        var isEmpty = fillCertsGridStore(KS.XCrypt.certGrid, certs, enableErrs);

        KS.XCrypt.certWin = KS.showModal(KS.XCrypt.certGrid, {
            title: KS.L10n.sign_ChooseCertificate + getSignerFilterString(''),
            autoHeight: false,
            height: 400,
            buttonAlign: 'left',
            buttons: [{
                xtype: 'label',
                hidden: KS.hideCertWindowBottomToolbarChooseCertificateLabel,
                text: KS.L10n.sign_ChooseCertificate + prepareCertIdForDisplay(ss.nastrKeyId)
            }, '->', {
                text: KS.L10n.sign_Choose,
                fn: fn,
                fnScope: scope,
                certs: certs,
                handler: certificateSelected
            }]
        }, true);

        if (isEmpty) {
            describeEmptyCertsList();
        }
    }

    function certificateSelected() {
        var recs = KS.XCrypt.certGrid.getSelectedRecs();
        if (!Ext.isEmpty(recs)) {
            KS.XCrypt.certWin.close();
            this.fn.call(this.fnScope, recs[0].data, this.certs);
        }
    }

    function describeEmptyCertsList() {
        var storedKeyId = KS.XCrypt.signSettings.nastrKeyId,
            hasUnfilteredCerts = true,
            certType = KS.isEmpty(storedKeyId) ? 'empty' : (storedKeyId[0] == '#' ? 'num' : name),
            genitive = (certType == 'num') ? (KS.L10n.sign_Number + prepareCertIdForDisplay(storedKeyId)) : (KS.L10n.sign_Name + storedKeyId),
            todo = KS.L10n.sign_IssueAdministrator;
        KS.XCrypt.signSettings.nastrKeyId = null;
        if (KS.isEmpty(KS.XCrypt.selectCertificateInternal)) {
            KS.showHtmlProtocol(KS.certNotFoundMsg.replace('{genitive}', genitive) + todo, KS.L10n.sign_DontFindCertificate);
            return;
        }
        KS.XCrypt.selectCertificateInternal(function (rawXml) {
            KS.XCrypt.signSettings.nastrKeyId = storedKeyId;
            try {
                var allCertsList = parseCertsXml(rawXml);
                hasUnfilteredCerts = !KS.isEmpty(allCertsList);
                if (hasUnfilteredCerts) {
                    KS.warning(KS.certNotFoundMsg.replace('{genitive}', genitive) + todo);
                    return;
                }
            } catch (e) {
            }
            if (certType == 'empty') {
                KS.warning(KS.L10n.sign_FromSystemDontInstalled);
            } else {
                KS.warning(KS.certNotFoundMsg.replace('{genitive}', genitive) + todo);
            }
        }, KS.XCrypt.signSettings.selectCertificateParams);
    }

    // Public
    KS.XCrypt = {
        pleaseWait: function (msg) {
            //KS.XCrypt.stopWaiting();
            if (KS.XCrypt.viewContext && KS.isFunction(KS.XCrypt.viewContext.pleaseWait)) {
                KS.XCrypt.viewContext.pleaseWait(msg);
            }
        },

        stopWaiting: function () {
            if (KS.XCrypt.viewContext && KS.isFunction(KS.XCrypt.viewContext.stopWaiting)) {
                KS.XCrypt.viewContext.stopWaiting();
            }
        },

        initialize: function (signSettings, fn, scope, mode) {
            if (signSettings) KS.XCrypt.signSettings = signSettings;
            switch (KS.edsType) {
                case 2:
                    var view = scope;
                    KS.wbe.initCryptoModule(view);
                    if (mode === 'login') {
                        KS.apply(Connection.Info, view.data);
                    }
                    fn.call(scope);
                    break;

                case 3:
                    KS.wbe.initEngine(scope);
                    if (!KS.XCrypt.cadesModule) {
                        KS.XCrypt.cadesModule = new CadesCryptoModule();
                    }
                    fn.call(scope);
                    break;
            }
        },

        selectCertificate: function (fn, scope, enableErrs) {
            KS.XCrypt.pleaseWait(KS.L10n.sign_AppealStoreCertificates);
            var ss = KS.XCrypt.signSettings;
            switch (KS.edsType) {
                case 2:
                    KS.XCrypt.cmInitor.container.Signer.SelectCertificates.call(KS.XCrypt.cmInitor,
                        {
                            keyId: ss.nastrKeyId,
                            issuerFilter: ss.issuerFilter,
                            filterOid: ss.filterOid,
                            numDaysWarnBeforeCertEnd: ss.numDaysWarnBeforeCertEnd + '',
                            onSuccess: function (certs) {
                                try {
                                    KS.XCrypt.stopWaiting();
                                    displayCertsList(certs, fn, scope, enableErrs);
                                } catch (e) {
                                    KS.XCrypt.stopWaiting();
                                    KS.showHtmlProtocol(e.message || e, KS.L10n.sign_ErrorReadListCertificates);
                                }
                            },
                            onError: function (err) {
                                KS.XCrypt.stopWaiting();
                                KS.showHtmlProtocol(err, KS.L10n.sign_ErrorReadListCertificates);
                            },
                            onCancel: function() {
                                KS.XCrypt.stopWaiting();
                            }
                        });
                    break;

                case 3:
                    var params = new CadesCertificateParams({
                        KeyId: ss.nastrKeyId,
                        FilterOid: ss.filterOid,
                        IssuerFilter: ss.issuerFilter,
                        SecretKeyValidity: ss.secretKeyValidity,
                        NumDaysWarnBeforeCertEnd: ss.numDaysWarnBeforeCertEnd + ''
                    });

                    KS.XCrypt.cadesModule.GetCertificates(params)
                        .then(function(certificates) {
                            try {
                                KS.XCrypt.stopWaiting();
                                displayCertsList(certificates, fn, scope, enableErrs);
                            } catch (e) { // Ошибка отображения списка сертификатов
                                KS.XCrypt.stopWaiting();
                                KS.showHtmlProtocol(e.message || e, KS.L10n.sign_ErrorReadListCertificates);
                            }
                        })
                        .catch(function (err) { // Ошибка чтения списка сертификатов или коннекта с плагином
                            KS.XCrypt.stopWaiting();
                            if (!(err.message && err.message.indexOf('#install#') > -1)) // #install# - показали окно установки
                                KS.showHtmlProtocol(err.message || err, KS.L10n.sign_ErrorReadListCertificates);
                        });
                    break;
            }
        },

        sign: function (text, cert, params, strParams, fn, scope) {
            KS.XCrypt.pleaseWait(KS.L10n.sign_SignatureBuilding);

            if (!KS.isEmpty(cert) && cert.charAt(0) !== '#')
                cert = '#' + cert;

            switch (KS.edsType) {
                case 2:
                    var signData = (params == 1)
                        ? KS.XCrypt.cmInitor.container.Signer.Detached.B64Data
                        : KS.XCrypt.cmInitor.container.Signer.Detached.Data;

                    signData.Sign({
                        data: text,
                        params: params,
                        signParams: {
                            //CertInclude : null
                        },
                        certSubj: cert,
                        onSuccess: function (hash) {
                            try {
                                KS.XCrypt.stopWaiting();
                                fn.call(scope, hash);
                            } catch (e) {
                                KS.XCrypt.stopWaiting();
                                KS.showHtmlProtocol(e.message || e, KS.L10n.sign_ErrorReadListCertificates);
                            }
                        },
                        onError: function (err) {
                            KS.showHtmlProtocol(err, KS.L10n.sign_ErrorMarked);
                        }
                    });

                    break;

                case 3:
                    KS.XCrypt.cadesModule.SignText(text, cert)
                        .then(function(hash) {
                            try {
                                KS.XCrypt.stopWaiting();
                                fn.call(scope, hash);
                            } catch (e) {
                                KS.XCrypt.stopWaiting();
                                KS.showHtmlProtocol(e.message || e, KS.L10n.sign_ErrorReadListCertificates);
                            }
                        })
                        .catch(function (err) {
                            KS.XCrypt.stopWaiting();
                            KS.showHtmlProtocol(err.message, KS.L10n.sign_ErrorMarked);
                        });
                    break;
            }
        },

        certInfo: function (cert, fn, scope) {
            if (KS.isEmpty(cert)) {
                fn.call(scope, {});
                return;
            }

            KS.XCrypt.pleaseWait(KS.L10n.sign_CheckedCertificate);
            switch (KS.edsType){
                case 2:
                    var signData = KS.XCrypt.cmInitor.container.Signer.Detached.Data;
                    signData.CertInfo({
                        certId: cert.replace('#', '').replace('?', '').replace(/ /g, ''),
                        onSuccess: function (certInfo) {
                            try {
                                KS.XCrypt.stopWaiting();
                                fn.call(scope, certInfo);
                            } catch (e) {
                            }
                        },
                        onError: function () {
                            KS.XCrypt.stopWaiting();
                            fn.call(scope, {});
                        }
                    });
                    break;

                default:
                    KS.XCrypt.stopWaiting();
                    fn.call(scope, {});
                    break;
            }            
        },

        signBinary: function(data, cert, params, strParams, fn, scope) {
            if (!KS.isEmpty(cert) && cert.charAt(0) !== '#')
                cert = '#' + cert;
            
            switch (KS.edsType) {
                case 2:
                    var signData = (params == 1)
                    ? KS.XCrypt.cmInitor.container.Signer.Detached.B64Data
                    : KS.XCrypt.cmInitor.container.Signer.Detached.Data;

                    signData.SignBinary({
                        data: data,
                        params: params,
                        signParams: {
                            CertInclude : 0
                        },
                        certSubj: cert,
                        onSuccess: function (hash) {
                            try {
                                KS.XCrypt.stopWaiting();
                                fn.call(scope, hash);
                            } catch (e) {
                                KS.XCrypt.stopWaiting();
                                KS.showHtmlProtocol(e.message || e, KS.L10n.sign_ErrorReadListCertificates);
                            }
                        },
                        onError: function (err) {
                            KS.showHtmlProtocol(err, KS.L10n.sign_ErrorMarked);
                        }
                    });
                    break;
                case 3:
                    new Promise((resolve) => {
                        const fr = new FileReader();

                        fr.onload = (e) => resolve(e.target.result);

                        fr.onerror = () => resolve(null);

                        fr.readAsDataURL(data);
                    }).then((text) => {
                        text = text.replace(/^data:[^,]+,/, '')
                        KS.XCrypt.cadesModule.SignText(text, cert)
                            .then(function (hash) {
                                try {
                                    KS.XCrypt.stopWaiting();
                                    fn.call(scope, hash);
                                } catch (e) {
                                    KS.XCrypt.stopWaiting();
                                    KS.showHtmlProtocol(e.message || e, KS.L10n.sign_ErrorReadListCertificates);
                                }
                            })
                            .catch(function (err) {
                                KS.XCrypt.stopWaiting();
                                KS.showHtmlProtocol(err.message, KS.L10n.sign_ErrorMarked);
                            });
                    });
                    break;
            }
        },

        signFileWeb: function (path, cert, params, strParams, fn, scope) {
            switch (KS.edsType) {
                case 2:
                case 3:
                    KS.error(KS.L10n.sign_FunctionDontSupported);
                    break;
            }
        },

        downloadFile: function (url, fn, scope) {
            switch (KS.edsType) {
                case 2:
                case 3:
                    KS.error(KS.L10n.sign_FunctionDontSupported);
                    break;
            }
        },

        certGridRenderer: function (value, metadata, record) {
            var error = getCertError(record);
            if (!KS.isEmpty(error)) {
                setTooltip(metadata, error);
            }
            return value;
        }
    }
})();
