﻿// ============= VIEW =======================
SelectionEditView = KS.extend(BaseSvodView, {
    constructor: function (viewId) {
        SelectionEditView.superclass.constructor.call(this, viewId);
    },

    onTemplateRendered: function () {
        if (this.data.identifier && this.data.identifier.IsNewOutcome) 
            this.touch();
        this.editableAnalyticFields = this.getEditableAnalyticFields();
        this.checkedFormsTree = [];
        this.getPeriodTypeVisibility(this.periodType.getValue());
    }
});

// ============= BASE =======================
(function (viewClass) {
    KS.apply(viewClass, {

        onSetSavedState: function(saved) {
            this.getToolbarItem(this.rootPanel, null, "SAVE").setDisabled(saved);
        },

        changedValues: function (change) {
            var view = this.parentView || this;
            if (view.hasChanges || change) 
                view.touch();
        },

        disable: function () {
            this.getToolbar(this.formsTreePanel).disable();
            this.getToolbar(this.analyticsTreeOuterPanel).disable();
            this.getToolbar(this.rowsGrid).disable();
            this.getToolbar(this.colsGrid).disable();
        },

        enable: function () {
            this.getToolbar(this.formsTreePanel).enable();
            this.getToolbar(this.analyticsTreeOuterPanel).enable();
            this.getToolbar(this.rowsGrid).enable();
            this.getToolbar(this.colsGrid).enable();
        },

        setRowColTotal: function () {
            var props = this.data.properties;
            this.getToolbarItem(this.rowsGrid, null, 'addRowTotal').setValue(props.AddRowTotal);
            this.getToolbarItem(this.colsGrid, null, 'addColTotal').setValue(props.AddColTotal);
        },

        setValues: function () {
            var props = this.data.properties;

            this.formsTree.getRootNode().removeAll(true);
            this.formsTree.fullReload();

            this.name.setValue(props.Name);
            this.note.setValue(props.Note);
            this.containerPanel.setTitle(props.Name);

            this.periodType.setValue(props.PeriodType);

            this.startPeriod.setType(props.StartPeriodType);
            if (this.startPeriod.getYear() !== props.StartYear) {
                this.startPeriod.setYear(props.StartYear);
            }
            this.startPeriod.setNumber(props.StartPeriodNumber);

            this.endPeriod.setType(props.EndPeriodType);
            if (this.endPeriod.getYear() !== props.EndYear) {
                this.endPeriod.setYear(props.EndYear);
            }
            this.endPeriod.setNumber(props.EndPeriodNumber);

            this.getPeriodTypeVisibility(props.PeriodType);
            this.setRowColTotal();
            this.reloadAnalytics();
        },

        collectValues: function () {
            var view = this,
                props = this.data.properties,
                name = this.name.getValue(),
                note = this.note.getValue();

            this.analyticRecords = [];
            this.rowsGrid.store.each(view.fillAnalyticRecord, view);
            this.colsGrid.store.each(view.fillAnalyticRecord, view);

            props.Name = Ext.isEmpty(name) ? 'Новая выборка' : name;
            props.Note = note;

            props.PeriodType = this.periodType.getValue();

            props.StartPeriodType = this.startPeriod.getType();
            props.StartYear = this.startPeriod.getYear();
            props.StartPeriodNumber = this.startPeriod.getNumber();

            props.EndPeriodType = this.endPeriod.getType();
            props.EndYear = this.endPeriod.getYear();
            props.EndPeriodNumber = this.endPeriod.getNumber();

            props.AddRowTotal = this.getToolbarItem(this.rowsGrid, null, 'addRowTotal').getValue();
            props.AddColTotal = this.getToolbarItem(this.colsGrid, null, 'addColTotal').getValue();
            
            return true;
        },

        onBeforeClose: function () {
            var view = this;
            if (this.hasChanges) {
                Ext.MessageBox.show({
                    icon: Ext.MessageBox.QUESTION,
                    title: 'Сохранить изменения?',
                    msg: 'Форма содержит несохраненные изменения. <br />Сохранить перед закрытием?',
                    buttons: Ext.MessageBox.YESNOCANCEL,
                    fn: function (btn) {
                        switch (btn) {
                            case 'yes':
                                view.saveChanges(true);
                                break;
                            case 'no':
                                view.close();
                                break;
                        }
                    }
                });
                return false;
            }
            return SelectionEditView.superclass.onBeforeClose.call(view);
        },

        onValuesSelected: function () {
            this.touch();
            this.reloadAnalytics();
        }
    });
}(SelectionEditView.prototype));

// ============= FORMS TREE =======================
(function (viewClass) {
    KS.apply(viewClass, {

        setStatusButtonState: function (btn) {
            var view = btn.parentView,
                selectedStatusLinks = [],
                notSelectedStatusLinks = [];
            view.getToolbar(view.formsTree).items.each(function (item) {
                if (item.link >= 0) {
                    if (item.pressed)
                        selectedStatusLinks.push(item.link);
                    else
                        notSelectedStatusLinks.push(item.link);
                }
            });
            view.disable();
            view.serverCall({
                method: 'SetStatusButtonState',
                params: [selectedStatusLinks, notSelectedStatusLinks],
                disableFog: true,
                success: view.touch,
                complete: view.enable
            });
        },

        setFormStatusButtons: function(selectedStatuses){
            var selLinks = [];
            Ext.each(selectedStatuses, function(status){
                if (status["LINK_SAVED_FORM_STATUS"])
                    selLinks.push(status["LINK_SAVED_FORM_STATUS"]);
            });
            var formsTreeTbar = this.getToolbar(this.formsTree, "top");
            if (formsTreeTbar && formsTreeTbar.items.items)
                Ext.each(formsTreeTbar.items.items, function(btn){
                    if (btn.isStatusBtn)
                        btn.setPressed(selLinks.indexOf(btn.link.toString()) > -1);
                })
        },

        formsTreeCheckChange: function (node, checked) {
            this.checkedFormsTree.push({ NodeLink : node.id, Checked : checked });
            if (node.hasChildNodes()) {
                this.checkAllSubnodes(node, checked);
                node.expand();
            }
            var view = this;
            this.serverCall({
                method: 'FormsTreeCheckChange',
                params: [this.checkedFormsTree],
                waitMessage: 'Обновление аналитических признаков',
                success: function() {
                    view.reloadAnalytics();
                    view.checkedFormsTree = [];
                }
            });
            this.touch();
        },

        formsTreeCheckChangeMin: function (node, checked) {
            this.checkedFormsTree.push({ NodeLink : node.id, Checked : checked });
            if (node.hasChildNodes()) {
                this.checkAllSubnodes(node, checked);
                node.expand();
            }
        },

        checkAllSubnodes: function (node, checked) {
            var view = this;
            Ext.each(node.childNodes, function (childNode) {
                if (childNode.data.checked !== checked) {
                    view.setChecked(childNode, checked);
                    view.formsTreeCheckChangeMin(childNode, checked);
                }
            });
        },

        setChecked: function (node, checked) {
            if (!Ext.isBoolean(checked)) 
                checked = !node.data.checked;
            node.set('checked', checked);
        },

        formsTreeNodeLoaded: function (root) {
            this.getToolbarItem(this.formsTreePanel, 'bottom', 'lbFormsCount').setText('Форм: ' + root.childNodes.length);
            if (this.selectedFormsNodeId){
                this.formsTree.setSelected(this.selectedFormsNodeId.toString());
                this.scrollToActiveNode(this.formsTree);
                this.selectedFormsNodeId = null;
            } else {
                this.formsTree.selectFirstNode();
            }
        },

        collapseFormsTree: function () {
            var view = this.parentView || this;
            view.formsTree.collapseAll();
        },

        expandFormsTree: function () {
            var view = this.parentView || this;
            view.formsTree.expandAll();
        },

        searchTree: 'formsTree',

        reloadFormsTree: function () {
            var view = this.parentView || this;
            if (view.startPeriod.getType() !== view.endPeriod.getType()) 
                return;
            
            if (view.isFormsReloading) {
                return;
            } else {
                view.isFormsReloading = true;
            }
            if (!view.collectValues()) {
                return;
            }
            view.touch();
            view.serverCall({
                method: 'SaveSelectionProperties',
                params: [view.data.properties],
                waitMessage: 'Отправка данных на сервер ...',
                success: view.doReloadFormsTree
            });
        },

        doReloadFormsTree: function () {
            var view = this;
            view.serverCall({
                method: 'ReloadFormsTree',
                params: [view.data.properties],
                waitMessage: 'Обновление форм ...',
                success: view.doReloadFormsTreeSucces
            });
        },

        doReloadFormsTreeSucces: function (data) {
            this.data = data;
            this.formsTree.getRootNode().removeAll(true);
            this.formsTree.fullReload();
            this.reloadAnalytics();
            this.isFormsReloading = false;
        },

        createFormColumn: function(){
            var view = this.parentView || this;
            view.getEditFormColumnPanel(false)
        },

        editFormColumn: function(){
            var view = this.parentView || this;
            view.getEditFormColumnPanel(true)
        },
        
        getEditFormColumnPanel: function(isEdit){
            var view = this;
            var selNodeId = view.formsTree.getSelNodeId();
            view.serverCall({
                method: 'GetEditFormColumnPanel',
                params: [isEdit, selNodeId],
                success: function(panel){
                    view.showEditFormColumnPanel(panel, isEdit);
                }
            });
        },

        showEditFormColumnPanel: function(panel, isEdit){
            if (Ext.isEmpty(panel)) return;
            this.editFormColumnPanel = this.createTemplateControl(panel);
            this.addPanelItems(panel, this.editFormColumnPanel);
            var view = this;
            this.editFormColumnWin = KS.showModal(this.editFormColumnPanel,
                {
                    title: 'Редактирование вычисляемого столбца',
                    autoHeight: false,
                    layout: "fit",
                    minHeight: 200,
                    height: Math.max(500, KS.rootViewport.getHeight() / 1.5),
                    minWidth: 300,
                    width: Math.max(700, KS.rootViewport.getWidth() / 2),
                    buttonAlign: 'left',
                    buttons: ['->',
                        {
                            text: 'ОК',
                            cls: 'dim-button',
                            handler: function() {
                                view.saveEditFormColumn(isEdit);
                            }
                        },
                        { xtype: 'tbspacer', width: 8 },
                        {
                            text: 'Отмена',
                            cls: 'dim-button',
                            handler: function() {
                                view.cancelEditFormColumn(isEdit);
                            }
                        }]
                },
                true);
        },

        saveEditFormColumn: function(isEdit){
            var view = this;
            var linkSelectCol = this.editFormColumnPanel.linkSelectCol;
            var code = this.formCodeEditor.getValue();
            var name = this.formNameEditor.getValue();
            var formula = this.formulaEditor.getValue();
            if (Ext.isEmpty(code) || Ext.isEmpty(name) || Ext.isEmpty(formula)){
                KS.alert("Код, название и формула не могут быть пустыми", "Внимание");
                return;
            }
            this.editFormColumnWin.mask('Сохранение...');
            view.serverCall({
                method: 'SaveEditFormColumn',
                params: [linkSelectCol, code, name, formula],
                success: function(newNodeName){
                    view.editFormColumnWin.unmask();
                    view.editFormColumnWin.close();
                    
                    if (isEdit) {
                        var selNode = view.formsTree.getSelNode();
                        if (selNode && newNodeName) {
                            selNode.set("text", newNodeName);
                            view.touch();
                        }
                    } else {
                        view.selectedFormsNodeId = linkSelectCol;
                        view.reloadFormsTree()
                    }
                },
                error: function(){
                    view.editFormColumnWin.unmask();
                }
            });
        },

        cancelEditFormColumn: function(isEdit){
            var view = this;
            var linkSelectCol = this.editFormColumnPanel.linkSelectCol;
            if (isEdit) {
                view.editFormColumnWin.close();
            } else {
                view.serverCall({
                    method: 'CancelEditFormColumn',
                    params: [linkSelectCol],
                    success: function () {
                        view.editFormColumnWin.unmask();
                        view.editFormColumnWin.close();
                    },
                    error: function () {
                        view.editFormColumnWin.unmask();
                    }
                });
            }
        },

        deleteFormColumn: function(){
            var view = this.parentView || this;
            var selNode = view.formsTree.getSelNode();
            if (selNode){
                view.serverCall({
                    method: 'DeleteFormColumn',
                    params: [selNode.id],
                    disableFog: true,
                    success: function (result) {
                        if (result){
                            view.removeFormNode(selNode)
                            view.touch();
                        }
                    }
                });
            }
        },
        
        removeFormNode: function(node){
            if (node.previousSibling)
                this.formsTree.getSelectionModel().select(node.previousSibling);
            node.remove();
        },

        // Пока только визуальное перемещение, без сохранения состояния. В толстом так же не работает сохранение на 16.05.2024
        moveFormsNode: function(){
            var view = this.parentView || this,
                code = this.tbarNode.code,
                isUp = code === "UP_FORM",
                selNode = view.formsTree.getSelNode();
            if (selNode && selNode.parentNode) {
                var parentNode = selNode.parentNode;
                if (parentNode.data.json.nodeDataType === 1 &&
                    parentNode.childNodes &&
                    parentNode.childNodes.length > 1) {
                    var nodeOrder = parentNode.indexOf(selNode);
                    if (isUp) {
                        if (!selNode.isFirst())
                            parentNode.insertChild(nodeOrder - 1, selNode);
                    } else {
                        if (!selNode.isLast()) {
                            parentNode.insertChild(nodeOrder + 2, selNode);
                        }
                    }
                }
            }
        },

        formulaEditorChange: function(comp, newValue){
            var view = this.parentView || this;
            view.serverCall({
                method: 'SetFormulaEditorValue',
                params: [newValue, comp.isFormEdit],
                disableFog: true,
                success: function(formulaReadOnlyValue){
                    if (view.formulaReadOnlyEditor)
                        view.formulaReadOnlyEditor.setValue(formulaReadOnlyValue);
                }
            });
        },

        columnGridAdd: function(){
            var view = this.parentView || this;
            if (view.formulaEditor && view.formColumnGrid){
                var selRows = view.formColumnGrid.getSelection();
                if (selRows && selRows.length > 0){
                    view.formulaEditor.setValue(view.formulaEditor.getValue() + "[" + KS.Grid.getAnyCase(selRows[0], "CODE") + "]")
                }
            }
        },

        columnGridDblClick: function(gridView, record){
            this.formulaEditor.setValue(this.formulaEditor.getValue() + "[" + KS.Grid.getAnyCase(record, "CODE") + "]")
        }
    });
}(SelectionEditView.prototype));

// ============= TOOLBAR HANDLERS =======================
(function (viewClass) {
    KS.apply(viewClass, {

        saveChanges: function (close) {
            var view = this.parentView || this;
            if (!view.collectValues()) return;
            if (view.data.isMySelection || view.data.isAllowEdit) {
                view.doSaveChanges(close, false);
            } else {
                KS.confirm("Вы не являетесь создателем данной выборки. Создать копию для Вас ?", "Подтвердите создание копии", function (btn) {
                    if (btn === 'yes') 
                        view.doSaveChanges(close, true);
                });
            }
        },

        doSaveChanges: function (close, makeCopy) {
            var view = this;
            view.serverCall({
                method: 'SaveSelectionChanges',
                params: [view.data.properties, makeCopy, view.analyticRecords],
                waitMessage: 'Сохранение ...',
                success: function (hasSaveErrors) {
                    if (hasSaveErrors === true) {
                        KS.alert('Документ не сохранен!');
                        return;
                    }
                    if (close === true) {
                        view.close();
                    } else {
                        view.doRefresh('Сохранение выполнено успешно');
                        view.swapCleanRefresh();
                    }
                }
            });
        },

        swapCleanRefresh: function () {
            this.getToolbarItem(this.rootPanel, null, 'CLEAN').setVisible(false);
            this.getToolbarItem(this.rootPanel, null, 'REFRESH').setVisible(true);
        },

        refresh: function () {
            var view = this.parentView || this;
            if (!view.hasChanges) {
                view.doRefresh();
                return;
            }
            KS.confirm("Обновить выборку? Несохраненные данные будут утеряны.", "Подтвердите обновление", function (btn) {
                if (btn === 'yes') 
                    view.doRefresh();
            });
        },

        clean: function () {
            var view = this.parentView || this;
            if (!view.hasChanges) {
                view.doRefresh(null, 'Выполнение ...');
                return;
            }
            KS.confirm("Очистить выборку? Несохраненные данные будут утеряны.", "Подтвердите очистку", function (btn) {
                if (btn === 'yes') 
                    view.doRefresh(null, 'Выполнение ...');
            });
        },

        doRefresh: function (alertMsg, waitMessage) {
            var view = this.parentView || this;
            view.serverCall({
                method: 'Reload',
                waitMessage: waitMessage || 'Обновление ...',
                success: function (data) {
                    view.data = data;
                    view.setValues();
                    view.discardChanges();
                    if (Ext.isString(alertMsg)) {
                        KS.alert(alertMsg);
                    }
                }
            });
        },

        execute: function () {
            var view = this.parentView || this;
            if (!view.collectValues()) return;

            var checkedFormLinks = [];
            Ext.each(view.formsTree.getChecked(), function (node) {
                checkedFormLinks.push(node.id);
            });

            view.serverCall({
                method: 'Execute',
                params: [view.data.properties, view.analyticRecords, checkedFormLinks],
                waitMessage: 'Печать выборки ...'
            });
        },

        helpBtn: function () {
            var href = "Docs/html/index.html";
            href += '?' + '2016' + '.htm';
            window.open(href, "_blank", "status=yes,location=yes,toolbar=yes,resizable=yes,menubar=yes,scrollbars=yes");
        }
    });
}(SelectionEditView.prototype));

// ============= SETTINGS =======================
(function (viewClass) {
    KS.apply(viewClass, {
        showSettings: function () {
            var view = this.parentView || this;
            view.serverCall({
                method: 'ShowSettings',
                waitMessage: 'Открывается ...',
                success: view.showSettingsWin
            });
        },

        showSettingsWin: function(settings) {
            var view = this;
            this.settingsGrid = this.getPropertyGridByOption(settings.List, "settingsGrid", settings.Name);
            this.settingsGrid.settings = settings;
            this.settingsWin = KS.showModal(this.settingsGrid,
            {
                layout: 'fit',
                plain: true,
                frame: true,
                autoScroll: true,
                modal: true,
                maximizable: true,
                minWidth: 400,
                minHeight: 300,
                width: Math.max(400, KS.rootViewport.getWidth() / 2),
                height: Math.max(300, KS.rootViewport.getHeight() / 1.5),
                title: "Свойства",
                buttons: ['->',
                    {
                        text: 'ОК',
                        cls: 'dim-button',
                        handler: function(){
                            view.saveSettings()
                        }
                    },
                    { xtype: 'tbspacer', width: 8 },
                    {
                        text: 'Отмена',
                        cls: 'dim-button',
                        handler: function(){
                            view.settingsWin.close()
                        }
                    }]
            }, true); 
        },

        saveSettings: function () {
            var view = this,
                list = view.settingsGrid.settings.List,
                source = view.settingsGrid.getSource();
            Ext.each(list, function(listItem, idx) {
                for (var property in source) {
                    if (source.hasOwnProperty(property)) {
                        if (listItem.Key === property.substring(3)) {
                            var value = view.settingsGrid.source[property];
                            switch (list[idx].Type) {
                                case 2: // Boolean
                                    list[idx].Value = value | 0;
                                    break;
                                case 4: // DropDownList
                                    list[idx].Value = list[idx].Items.indexOf(value);
                                    break;
                                default:
                                    list[idx].Value = value;
                                    break;
                            }
                        }
                    }
                }
            });
            view.settingsWin.mask('Сохранение настроек ...');
            view.serverCall({
                method: 'SaveSettings',
                waitMessage: 'Сохранение настроек ...',
                params: [view.settingsGrid.settings],
                success: function () {
                    view.settingsWin.unmask()
                    view.settingsWin.close();
                    view.touch();
                }
            });
        }
    });
}(SelectionEditView.prototype));

// ============= PERIODS =======================
(function (viewClass) {
    KS.apply(viewClass, {

        periodTypeChange: function (combo, newValue) {
            this.startPeriod.setNumber(1);
            this.startPeriod.setType(newValue);
            this.endPeriod.setNumber(1);
            this.endPeriod.setType(newValue);
            this.getPeriodTypeVisibility(newValue);
            this.touch();
        },

        periodChange: function () {
            this.reloadFormsTree();
        },

        changePeriodType: function (combo, rec) {
            var view = combo.parentView,
                periodType = rec.get('code');
            if (periodType === view.data.properties.PeriodType)
                return;
            view.data.properties.PeriodType = periodType;
        },
        
        getPeriodTypeVisibility: function (type) {
            switch (+type) {      
                case 2:
                case 4:
                    this.startPeriod.getComponent("type").setHidden(false);
                    this.endPeriod.getComponent("type").setHidden(false);
                    break;
                default:
                    this.startPeriod.getComponent("type").setHidden(true);
                    this.endPeriod.getComponent("type").setHidden(true);
                    break;
            }
        }
    });
}(SelectionEditView.prototype));

// ============= ANALYTICS =======================
(function (viewClass) {
    KS.apply(viewClass, {

        moveItem: function () {
            var view = this.parentView,
                srcGrid,
                srcTree,
                cmd;

            switch (this.direction) {
                case 0:
                    srcGrid = view.rowsGrid;
                    break;
                case 1:
                    srcGrid = view.colsGrid;
                    break;
                case 2:
                    srcTree = view.analyticsTree;
                    break;
                default:
                    KS.log('Unknown direction: ' + this.direction);
                    return;
            }

            switch (this.iconCls) {
                case 'ks-icon-up':
                    cmd = 'up';
                    break;
                case 'ks-icon-down':
                    cmd = 'down';
                    break;
                case 'ks-icon-delete':
                    cmd = 'del';
                    break;
                case 'ks-icon-arrows_down':
                    cmd = 'toCols';
                    break;
                case 'ks-icon-arrows_right':
                    cmd = 'toRows';
                    break;
                default:
                    KS.log('Unknown cmd: ' + this.icon);
                    return;
            }

            if (srcGrid) {
                var selection = srcGrid.getSelectionModel().selection;

                if (!selection) 
                    return;

                var recIdx = selection.rowIdx;
                var colIdx = selection.colIdx;

                var rec = selection.record,
                    total = srcGrid.store.getCount();

                if ((cmd === 'up' && recIdx <= 0) || (cmd === 'down' && recIdx >= total - 1))
                    return;

                view.activeLink = KS.Grid.getAnyCase(rec, 'link_analytic');
                view.activeColIdx = colIdx;
            } else if (srcTree) {
                var node = srcTree.getSelNode();
                if (!node) 
                    return;

                view.activeLink = node.id;
            } else {
                return;
            }

            if (!view.activeLink)
                return;

            view.disable();
            view.serverCall({
                method: 'MoveItem',
                params: [view.activeLink, cmd],
                disableFog: true,
                success: view.reloadAnalytics,
                error: view.enable
            });
            view.touch();
        },

        findAnyCase: function (store, field, value) {
            if (!store || Ext.isEmpty(field)) 
                return -1;
            var recIdx = store.find(field, value);
            if (recIdx < 0) 
                recIdx = store.find(field.toLowerCase(), value);
            if (recIdx < 0) 
                recIdx = store.find(field.toUpperCase(), value);
            return recIdx;
        },

        selectActiveLink: function () {  // afterjsonload у colsGrid , т.к. он обновляется последним
            if (!this.activeLink) {
                this.disableFilterBtnInGrid(this.rowsGrid);
                this.disableFilterBtnInGrid(this.colsGrid);
                return;
            }

            var recIdx;
            recIdx = this.findAnyCase(this.rowsGrid.store, 'link_analytic', this.activeLink);
            if (recIdx >= 0) {
                this.selectItemInGrid(this.rowsGrid, recIdx);
                this.disableFilterBtnInGrid(this.colsGrid);
                this.activeLink = null;
                return;
            }

            recIdx = this.findAnyCase(this.colsGrid.store, 'link_analytic', this.activeLink);
            if (recIdx >= 0) {
                this.selectItemInGrid(this.colsGrid, recIdx);
                this.disableFilterBtnInGrid(this.rowsGrid);
                this.activeLink = null;
                return;
            }

            this.analyticsTree.setSelected(this.activeLink);
            this.disableFilterBtnInGrid(this.rowsGrid);
            this.disableFilterBtnInGrid(this.colsGrid);
            this.activeLink = null;
        },
        
        findAnalyticRow: function(rowId){
            var record = this.rowsGrid.findRecord(rowId, this.rowsGrid.closeCode.toLowerCase()) ||
                         this.rowsGrid.findRecord(rowId, this.rowsGrid.closeCode.toUpperCase());
            if (Ext.isEmpty(record)) {
                record = this.colsGrid.findRecord(rowId, this.colsGrid.closeCode.toLowerCase()) ||
                         this.colsGrid.findRecord(rowId, this.colsGrid.closeCode.toUpperCase());
            }
            return record;
        },

        selectItemInGrid: function (grid, recIdx) {
            grid.getSelectionModel().select({
                row: recIdx,
                column: this.activeColIdx || 0
            });

            var record = grid.getStore().getAt(recIdx);
            this.setDisableFilterButtons(record, grid);
        },

        getEditableAnalyticFields: function () {
            var res = [];
            Ext.each(this.rowsGrid.getColumns(), function (colCfg) {
                if (colCfg.visibility && colCfg.dataIndex.substring(0,3) === "IS_") 
                    res.push(colCfg.dataIndex);
            });
            return res;
        },

        analyticsGridDblClick: function (gridView, rowIdx, cellIdx, record) {
            var srcGrid = gridView.grid;
            var column = srcGrid.getColCfgByIndex(cellIdx);
            var link = KS.Grid.getAnyCase(record, 'link_analytic');
            
            if (column.dataIndex.toUpperCase() === 'FILTER') {
                this.filter(link);
                return false;
            }
            
            this.getEditAnalyticColumnPanel(true, link, false);
            return (column.dataType !== 2);
        },

        analyticsGridClick: function (gridView, td, cellIndex, record, tr, rowIndex, e) {
            this.setDisableFilterButtons(record, gridView.grid);
            var dataIndex = e.position.column.dataIndex.toUpperCase();
            if (dataIndex === "IS_VISIBLE" || dataIndex === "IS_TOTAL"){
                this.setAnalyticsGridValue(record, dataIndex);
            }
        },

        setAnalyticsGridValue: function(record, dataIndex){
            var newValue = KS.Grid.getAnyCase(record, dataIndex);
            var link = KS.Grid.getAnyCase(record, 'link_analytic');
            var view = this;
            this.serverCall({
                method: 'SetAnalyticsGridValue',
                params: [link, dataIndex, newValue],
                disableFog: true,
                success: function(result){
                    if (result)
                        view.touch()
                }                    
            });
        },

        reloadAnalytics: function () {
            this.enable();

            this.analyticsTree.getRootNode().removeAll();
            this.analyticsTree.fullReload();

            this.rowsGrid.linkAnalytic = null;
            this.colsGrid.linkAnalytic = null;
            this.rowsGrid.reload();
            this.colsGrid.reload();
        },

        fillAnalyticRecord: function (rec) {
            var analyticLink = KS.Grid.getAnyCase(rec, 'link_analytic'),
                recArg = { Link: analyticLink };
            Ext.each(this.editableAnalyticFields, function (eaf) {
                var value = rec.get(eaf);
                if (Ext.isDefined(value)) 
                    recArg[eaf] = value;
            });
            this.analyticRecords.push(recArg);
        },

        totalCheckboxChange: function() {
            this.touch();
        },

        analyticGridDblClick: function(treeView, record){
            this.getEditAnalyticColumnPanel(true, record.id, true);
        },

        createAnalyticColumn: function(){
            var view = this.parentView || this;
            view.getEditAnalyticColumnPanel(false, null, true);
        },

        editAnalyticColumn: function(){
            var view = this.parentView || this;
            var selNodeId = view.analyticsTree.getSelNodeId();
            view.getEditAnalyticColumnPanel(true, selNodeId, true);
        },

        getEditAnalyticColumnPanel: function(isEdit, nodeId, editInTree){
            var view = this;
            view.serverCall({
                method: 'GetEditAnalyticColumnPanel',
                params: [isEdit, nodeId],
                success: function(panel){
                    view.showEditAnalyticColumnPanel(panel, isEdit, editInTree);
                }
            });
        },

        showEditAnalyticColumnPanel: function(panel, isEdit, editInTree){
            if (Ext.isEmpty(panel)) return;
            this.editAnalyticColumnPanel = this.createTemplateControl(panel);
            this.addPanelItems(panel, this.editAnalyticColumnPanel);
            var view = this;
            this.editAnalyticColumnWin = KS.showModal(this.editAnalyticColumnPanel,
                {
                    title: 'Редактирование вычисляемого столбца',
                    autoHeight: false,
                    layout: "fit",
                    minHeight: 200,
                    height: Math.max(500, KS.rootViewport.getHeight() / 1.5),
                    minWidth: 300,
                    width: Math.max(700, KS.rootViewport.getWidth() / 2),
                    buttonAlign: 'left',
                    buttons: ['->',
                        {
                            text: 'ОК',
                            cls: 'dim-button',
                            handler: function() {
                                view.saveEditAnalyticColumn(isEdit, editInTree);
                            }
                        },
                        { xtype: 'tbspacer', width: 8 },
                        {
                            text: 'Отмена',
                            cls: 'dim-button',
                            handler: function() {
                                view.cancelEditAnalyticColumn();
                            }
                        }]
                },
                true);
        },

        saveEditAnalyticColumn: function(isEdit, editInTree){
            var view = this;
            var selectedNodeId = this.editFormColumnPanel.selectedNodeId;
            var oldCode = this.editFormColumnPanel.oldCode;
            var newCode = this.formCodeEditor.getValue();
            var name = this.formNameEditor.getValue();
            var formula = this.formulaEditor.getValue();
            if (Ext.isEmpty(newCode) || Ext.isEmpty(name) || Ext.isEmpty(formula)){
                KS.alert("Код, название и формула не могут быть пустыми", "Внимание");
                return;
            }
            this.editAnalyticColumnWin.mask('Сохранение...');
            view.serverCall({
                method: 'SaveEditAnalyticColumn',
                params: [isEdit, oldCode, newCode, name, formula],
                success: function(success){
                    view.editAnalyticColumnWin.unmask();
                    if (success) {
                        view.editAnalyticColumnWin.close();
                        if (isEdit) {
                            if (editInTree) {
                                var selNode = view.analyticsTree.getSelNode();
                                if (selNode)
                                    selNode.set("text", name);
                            } else {
                                var rec = view.findAnalyticRow(selectedNodeId);
                                if (rec)
                                    rec.set("NAME", name);
                            }
                        } else {
                            view.reloadAnalytics();
                        }
                        view.touch();
                    }
                },
                error: function(){
                    view.editAnalyticColumnWin.unmask();
                }
            });
        },
        
        cancelEditAnalyticColumn: function(){
            this.editAnalyticColumnWin.close();
        },

        deleteAnalyticColumn: function(){
            var view = this.parentView || this;
            var selNode = view.analyticsTree.getSelNode();
            if (selNode){
                view.serverCall({
                    method: 'DeleteAnalyticColumn',
                    params: [selNode.id],
                    disableFog: true,
                    success: function (result) {
                        if (result){
                            view.removeAnalyticNode(selNode)
                            view.touch();
                        }
                    }
                });
            }
        },

        removeAnalyticNode: function(node){
            if (node.previousSibling)
                this.analyticsTree.getSelectionModel().select(node.previousSibling);
            node.remove();
        }
    });
}(SelectionEditView.prototype));

// ============= FILTERS =======================
(function (viewClass) {
    KS.apply(viewClass, {

        filter: function (link) {
            var view = this.parentView || this;
            if ((typeof link) === "object") 
                link = view.getActiveSelectionLink(this);
            
            view.activeLink = link;
            view.activeColIdx = view.getActiveSelectionColIdx(this);
            view.serverCall({
                method: 'SetFilter',
                params: [view.activeLink],
                waitMessage: 'Установка фильтра ...',
                success: view.showInternalFilter
            });
        },

        dictFilter: function () {
            var view = this.parentView;
            view.activeLink = view.getActiveSelectionLink(this);
            view.activeColIdx = view.getActiveSelectionColIdx(this);
            view.serverCall({
                method: 'OpenCellFilterDict',
                params: [view.activeLink],
                waitMessage: 'Установка фильтра ...',
                success: view.showInternalFilter
            });
        },

        showInternalFilter: function (filterGrid) {
            if (!filterGrid)
                return;
            var innerFilterGrid = {
                title: filterGrid.json.Caption,
                additional: filterGrid.json,
                autoHeight: false,
                height: 400,
                buttonalign: 'right',
                buttons: new Array({
                    text: 'Сохранить',
                    scope: this,
                    handler: this.saveInternalFilter
                })
            };
            this.filterGrid = KS.create(filterGrid);
            this.filterWin = KS.showModal(this.filterGrid, innerFilterGrid);
        },

        saveInternalFilter: function () {
            var cp = this.getCheckedInternalFilterRows();
            if (Ext.isEmpty(cp))
                return;
            var constantCode = this.filterWin.additional['CONSTANT_CODE'] || null,
                isAdditional = this.filterWin.additional['IS_ADDITIONAL'] || false,
                attributeLink = this.filterWin.additional['ATTRIBUTE_LINK'] || -1;
            this.filterWin.close();
            var view = this;
            this.serverCall({
                method: 'SetInternalFilter',
                waitMessage: 'Сохранение ...',
                params: [cp, this.activeLink, constantCode, isAdditional, attributeLink],
                success: function() {
                    view.touch();
                    if (constantCode === "SAVED_FORM_STATUS_NAME")
                        view.setFormStatusButtons(cp);
                    view.reloadAnalytics();
                }
            });
        },

        getCheckedInternalFilterRows: function () {
            var view = this,
                recs = view.filterGrid.getSelectedRows(),
                res = [];
            Ext.each(recs, function (rec) {
                var entry = {};
                Ext.each(view.filterWin.additional['KEYS'], function (key) {
                    entry[key] = KS.Grid.getAnyCase(rec, key);
                });
                res.push(entry);
            });
            return res;
        },

        removeFilter: function () {
            var view = this.parentView;
            view.activeLink = view.getActiveSelectionLink(this);
            view.activeColIdx = view.getActiveSelectionColIdx(this);
            view.serverCall({
                method: 'RemoveFilter',
                params: [view.activeLink],
                waitMessage: 'Очистка фильтра ...',
                success: function(){
                    view.reloadAnalytics();
                    view.touch();
                }
            });
        },

        editFilter: function(){
            var view = this.parentView;
            view.activeLink = view.getActiveSelectionLink(this);
            view.activeColIdx = view.getActiveSelectionColIdx(this);
            view.serverCall({
                method: 'GetAnalyticUserFilter',
                params: [view.activeLink],
                success: view.showUserFilterWin
            });
        },

        showUserFilterWin: function(userFilterValue){
            var view = this;
            view.userFilterWin = KS.showModal({
                xtype: 'textarea',
                itemId: 'userFilterArea',
                value: userFilterValue
            },{
                layout: 'fit',
                plain: true,
                frame: true,
                autoScroll: true,
                modal: true,
                maximizable: true,
                minWidth: 400,
                minHeight: 300,
                width: Math.max(400, KS.rootViewport.getWidth() / 2),
                height: Math.max(300, KS.rootViewport.getHeight() / 1.5),
                title: 'Пользовательский фильтр',
                buttons: ['->', 
                        {
                            text: 'ОК',
                            cls: 'dim-button',
                            handler: function () {
                                var newValue = view.userFilterWin.getComponent("userFilterArea").getValue();
                                view.setUserFilter(newValue);
                            }
                        }, 
                        { xtype: 'tbspacer', width: 8 },
                        {
                            text: 'Отмена',
                            cls: 'dim-button',
                            handler: function() {
                                view.userFilterWin.close();
                            }
                        }]
            }, true);
        },
        
        setUserFilter: function(newValue){
            var view = this;
            view.userFilterWin.close();
            view.serverCall({
                method: 'SetUserFilter',
                params: [view.activeLink, newValue],
                success: function(result) {
                    if (result) {
                        view.reloadAnalytics();
                        view.touch();
                    }
                }
            });
        },

        setDisableFilterButtons: function(record, grid){
            var view = this;
            
            var linkAnalytic = KS.Grid.getAnyCase(record, "link_analytic");
            if (grid.linkAnalytic === linkAnalytic) return; // чтобы не обращаться лишний раз к серверу, если кликаем по одной и той же строке
            grid.linkAnalytic = linkAnalytic;
            
            var rowData = {
                "TYPE" : KS.Grid.getAnyCase(record, "TYPE"),
                "LINK_CONSTANT" : KS.Grid.getAnyCase(record, "LINK_CONSTANT"),
                "LINK_SELECT_COL" : KS.Grid.getAnyCase(record, "LINK_SELECT_COL")
            }
            this.serverCall({
                method: 'GetDisableFilterButton',
                params: [rowData],
                disableFog: true,
                success: function(buttons) {
                    view.setFilterButtonsState(buttons, grid)
                }
            });
        },

        setFilterButtonsState:function(buttons, grid){
            if (Ext.isEmpty(buttons)) return;
            for (var buttonCode in buttons) {
                if (buttons.hasOwnProperty(buttonCode)) {
                    var btn = this.getToolbarItem(grid, null, buttonCode);
                    if (btn) {
                        btn.setDisabled(buttons[buttonCode])
                        if (!buttons[buttonCode])
                            btn.el.dom.setAttribute("data-qtip", btn.tooltipEl.getAttribute("data-qtip")); // костыльное возвращение tooltip, т.к. они теряются после disable/enable
                    }
                }
            }
        },
        
        disableFilterBtnInGrid: function(grid) {
            var buttonsState = {
                "CLEAN_FILTER": true,
                "DICT_FILTER": true,
                "FILTER": true,
                "EDIT_FILTER": true
            }
            this.setFilterButtonsState(buttonsState, grid)
        },
        
        getActiveSelectionLink: function (btn) {
            var selection = this.getActiveSelection(btn)
            if (selection) 
                return KS.Grid.getAnyCase(selection.record, 'link_analytic');
            return null;
        },

        getActiveSelectionColIdx: function (btn) {
            var selection = this.getActiveSelection(btn)
            if (selection)
                return selection.colIdx;
            return null;
        },
        
        getActiveSelection: function(btn) {
            var srcGrid;
            if (!btn)
                return null;

            switch (btn.direction) {
                case 0:
                    srcGrid = this.rowsGrid;
                    break;
                case 1:
                    srcGrid = this.colsGrid;
                    break;
                default:
                    KS.log('Unsupported direction: ' + this.direction);
                    break;
            }
            if (srcGrid)
                return srcGrid.getSelectionModel().selection;
        },

        changeAdditionalCell:function(oper, closeCode, gridId){
            var grid = this[gridId];
            if (grid){
                var record = grid.findRecord(closeCode, grid.closeCode.toLowerCase()) ||
                             grid.findRecord(closeCode, grid.closeCode.toUpperCase());
                if (record){
                    var view = this;
                    var analyticLink = KS.Grid.getAnyCase(record, 'link_analytic');
                    if (oper == "edit"){
                        view.openEditAdditionalCell(analyticLink, record);
                        return;
                    }
                    grid.mask();
                    view.serverCall({
                        method: 'ChangeAdditionalCell',
                        params: [oper, analyticLink],
                        disableFog: true,
                        success: function(result) {
                            grid.unmask();
                            if (result.success) {
                                record.set("ADDITIONAL", result.newValue);
                                view.touch();
                            }
                        },
                        error: function(){
                            grid.unmask();
                        }
                    });
                }
            }
        },

        openEditAdditionalCell: function(analyticLink, record){
            var view = this;
            var oldValue = KS.Grid.getAnyCase(record, "ADDITIONAL");
            view.additionalCellWin = KS.showModal({
                xtype: 'textarea',
                itemId: 'additionalCellArea',
                value: oldValue
            },{
                layout: 'fit',
                plain: true,
                frame: true,
                autoScroll: true,
                modal: true,
                maximizable: true,
                minWidth: 300,
                minHeight: 200,
                width: Math.min(300, KS.rootViewport.getWidth() / 2),
                height: Math.min(200, KS.rootViewport.getHeight() / 1.5),
                title: 'Редактирование...',
                buttonAlign: 'right',
                buttons: ['->', {
                        text: 'ОК',
                        cls: 'dim-button',
                        handler: function () {
                            var newValue = view.additionalCellWin.getComponent("additionalCellArea").getValue();
                            view.setAdditionalCellValue(analyticLink, record, newValue);
                            view.additionalCellWin.close(true);
                        }
                    }, 
                    { xtype: 'tbspacer', width: 8 },
                    {
                        text: 'Отмена',
                        cls: 'dim-button',
                        handler: function () {
                            view.additionalCellWin.close();
                        }
                    }]
            }, true);
        },
        
        setAdditionalCellValue: function(analyticLink, record, newValue){
            var view = this,
                grid = record.store.grid;
            grid.mask();
            view.serverCall({
                method: 'SetAdditionalCellValue',
                params: [analyticLink, newValue],
                disableFog: true,
                success: function(result) {
                    grid.unmask();
                    if (result.success) {
                        record.set("ADDITIONAL", result.newValue);
                        view.touch();
                    }
                },
                error: function(){
                    grid.unmask();
                }
            });
        },

        changeAdditionalButtons: function(grid, buttons, value, record) {
            var type = Number(KS.Grid.getAnyCase(record, 'TYPE'));
            switch (type) {
                case 0:
                    var linkConstant = KS.Grid.getAnyCase(record, 'LINK_CONSTANT');
                    if (!this.data.analyticsConstants)
                        return value;

                    var isFilter = this.data.analyticsConstants.find((element) => element.link_constant.toString() === linkConstant).is_filter;

                    if (isFilter)
                        return buttons.cleanBtn + value + buttons.addBtn + buttons.dictBtn + buttons.editBtn;
                    else
                        return buttons.cleanBtn + value + buttons.addBtn.replace('32px', '16px') + buttons.editBtn;
                    
                case 1:
                    var linkSelect = KS.Grid.getAnyCase(record, 'LINK_SELECT_COL');
                    var dataType = this.data.nodeTypes[linkSelect];

                    if (dataType === 6)
                        return buttons.cleanBtn + value + buttons.addBtn.replace('32px', '16px') + buttons.editBtn;
                    else
                        return buttons.cleanBtn + value + buttons.addBtn + buttons.dictBtn + buttons.editBtn;

                default:
                    return buttons.cleanBtn + value + buttons.addBtn + buttons.dictBtn + buttons.editBtn;
            }
        },

        changeFilterCell: function(oper, closeCode, gridId){
            var grid = this[gridId];
            if (grid){
                var record = grid.findRecord(closeCode, grid.closeCode.toLowerCase()) ||
                    grid.findRecord(closeCode, grid.closeCode.toUpperCase());
                if (record){
                    var view = this;
                    view.activeLink = KS.Grid.getAnyCase(record, 'link_analytic');
                    if (oper === 'edit'){
                        view.serverCall({
                            method: 'GetAnalyticUserFilter',
                            params: [view.activeLink],
                            success: view.showUserFilterWin
                        });
                        return;
                    }
                    if (oper === 'clean'){
                        view.serverCall({
                            method: 'RemoveFilter',
                            params: [view.activeLink],
                            waitMessage: 'Очистка фильтра ...',
                            success: function(){
                                view.reloadAnalytics();
                                view.touch();
                            }
                        });
                    }
                    if (oper === 'dict'){
                        view.serverCall({
                            method: 'OpenCellFilterDict',
                            params: [view.activeLink],
                            waitMessage: 'Установка фильтра ...',
                            success: view.showInternalFilter
                        });
                    }
                }
            }
        },

        changeFilterButtons: function(grid, buttons, value, record) {
            var type = Number(KS.Grid.getAnyCase(record, 'TYPE'));
            switch (type) {
                case 0:
                    var linkConstant = KS.Grid.getAnyCase(record, 'LINK_CONSTANT');
                    if (!this.data.analyticsConstants)
                        return value;

                    var rowData = this.data.analyticsConstants.find((element) => element.link_constant.toString() === linkConstant);

                    if (rowData.is_filter) {
                        switch (rowData.code) {
                            case 'TASK_DATE':
                            case 'SAVED_FORM_STATUS_NAME':
                            case 'ORGANIZATION_CODE':
                            case 'ORGANIZATION_NAME':
                            case 'PARENT_ORGANIZATION_CODE':
                            case 'PARENT_ORGANIZATION_NAME':
                            case "FOUNDER_ORGANIZATION_NAME":
                                return buttons.cleanBtn + value + buttons.dictBtn.replace('16px', '0');

                            case 'SIDEWALL_CODE':
                            case 'SIDEWALL_CODE_2':
                                return buttons.cleanBtn + value + buttons.editBtn;

                            default:
                                return buttons.cleanBtn + value;
                        }
                    }
                    else
                        return value;
    
                case 1:
                    var linkSelect = KS.Grid.getAnyCase(record, 'LINK_SELECT_COL');
                    var dataType = this.data.nodeTypes[linkSelect];
                    
                    if (dataType === 6)
                        return buttons.cleanBtn + value + buttons.editBtn;
                    else
                        return buttons.cleanBtn + value + buttons.dictBtn + buttons.editBtn;
    
                default:
                    return buttons.cleanBtn + value;
            }
        }
    });
}(SelectionEditView.prototype));

// ============= SORTERS =====================
(function(viewClass) {
    KS.apply(viewClass, {
        sortAnalyticsTree: function () {
            var view = this.parentView;
            var newDirection, directionIcon = 'sort_up';

            var store = view.analyticsTree.getStore();
            store.remoteSort = false;
            var sorters = store.getSorters();

            if (Ext.isEmpty(sorters.items)) {
                sorters.add({
                    property: 'text',
                    direction: 'ASC'
                });
            } else {
                var currentDirection = sorters.items[0].getDirection();

                switch (currentDirection) {
                case 'ASC':
                    newDirection = 'DESC';
                    directionIcon = 'sort_down';
                    break;
                case 'DESC':
                    newDirection = 'ASC';
                    directionIcon = 'sort_up';
                    break;
                }

                sorters.clear();

                sorters.add({
                    property: 'text',
                    direction: newDirection || 'ASC'
                });
            }
            view.setSortDirectionIcon(directionIcon);
        },

        setSortDirectionIcon: function (icon) {
            var view = this;
            var button = view.getToolbarItem(view.analyticsTree, null, 'ICON_SORT');
            button.setIconCls('ks-icon-' + icon);
        }
    });
}(SelectionEditView.prototype));

function changeAdditionalCell(viewId, oper, closeCode, gridId) {
    var view = KS.getView(viewId);
    if (view)
        view.changeAdditionalCell(oper, closeCode, gridId);
}

function selectionAdditionalRenderer(value, metadata, record, rowIndex, colIndex, store) {
    var grid = store.grid,
        closeCode = KS.Grid.getAnyCase(record, grid.closeCode),
        viewId = grid.parentView.viewID,
        rightBtnStyle = 'vertical-align: middle; position: absolute; top: 0; bottom: 0; margin: auto 0;',
        cleanBtn = '<img src="PlatformHandler.axd?icon=clean.png" onclick="changeAdditionalCell(\'' + viewId + '\',\'clean\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="vertical-align: middle;"/>',
        addBtn = '<img src="PlatformHandler.axd?icon=add.png" onclick="changeAdditionalCell(\'' + viewId + '\',\'add\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="' + rightBtnStyle + 'right: 32px;' + '"/>',
        dictBtn = '<img src="PlatformHandler.axd?icon=dict.png" onclick="changeAdditionalCell(\'' + viewId + '\',\'dict\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="' + rightBtnStyle + 'right: 16px;' + '"/>',
        editBtn = '<img src="PlatformHandler.axd?icon=edit.png" onclick="changeAdditionalCell(\'' + viewId + '\',\'edit\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="' + rightBtnStyle + 'right: 0;' + '"/>';

    var view = KS.getView(viewId);
    if (!view)
        return;

    var buttons = {
        cleanBtn : cleanBtn,
        addBtn : addBtn,
        dictBtn : dictBtn,
        editBtn : editBtn
    }
    return view.changeAdditionalButtons(grid, buttons, value, record);
}

function changeFilterCell(viewId, oper, closeCode, gridId) {
    var view = KS.getView(viewId);
    if (view)
        view.changeFilterCell(oper, closeCode, gridId);
}

function selectionFilterRenderer(value, metadata, record, rowIndex, colIndex, store) {
    var grid = store.grid,
        closeCode = KS.Grid.getAnyCase(record, grid.closeCode),
        viewId = grid.parentView.viewID,
        rightBtnStyle = 'vertical-align: middle; position: absolute; top: 0; bottom: 0; margin: auto 0;',
        cleanBtn = '<img src="PlatformHandler.axd?icon=clean.png" onclick="changeFilterCell(\'' + viewId + '\',\'clean\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="vertical-align: middle;"/>',
        dictBtn = '<img src="PlatformHandler.axd?icon=dict.png" onclick="changeFilterCell(\'' + viewId + '\',\'dict\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="' + rightBtnStyle + 'right: 16px;' + '"/>',
        editBtn = '<img src="PlatformHandler.axd?icon=edit.png" onclick="changeFilterCell(\'' + viewId + '\',\'edit\',\'' + closeCode + '\',\'' + grid.ctrlId +'\')" style="' + rightBtnStyle + 'right: 0;' + '"/>';

    var view = KS.getView(viewId);
    if (!view)
        return;

    var buttons = {
        cleanBtn : cleanBtn,
        dictBtn : dictBtn,
        editBtn : editBtn
    }
    return view.changeFilterButtons(grid, buttons, value, record);
}

if (typeof (Sys) !== 'undefined') { Sys.Application.notifyScriptLoaded(); }