using System;
using System.Collections.Generic;
using System.Text;
using System.Threading.Tasks;
using Keysystems.Core.Serialization.Json.JavaScriptObjects;
using Keysystems.Core.Web.Common;
using Keysystems.Core.Web.Controls;
#if NETCOREAPP
using System.Linq;
using Keysystems.Core.Web.Utils;
using Keysystems.Core.Web.Views;
#else
using System.IO;
using System.Web.UI;
using Keysystems.Core.Native.Cross;
using Keysystems.Core.Web.Http;
#endif

namespace Keysystems.Core.Web.Scripts
{
    /// <summary>
    /// </summary>
    public class PlatformScriptManager
#if NETCOREAPP
        : Control
#else
        : ScriptManager
#endif
    {
        #region Private fields

        /// <summary>
        ///   Список уровней
        /// </summary>
        private readonly List<int> m_includeLevels = new List<int>();

        /// <summary>
        ///   Список подключаемых файлов
        /// </summary>
        private readonly Dictionary<string, ScriptIncludeReference> m_scriptIncludes =
            new Dictionary<string, ScriptIncludeReference>();

        /// <summary>
        /// </summary>
        private class ScriptIncludeReference
        {
            /// <summary>
            /// </summary>
            public Type Type { get; set; }

            /// <summary>
            ///   Имя ресурса или URL
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            ///   Порядок регистрации
            /// </summary>
            public int Order { get; set; }
        }

        #endregion

        #region Уровни (порядок) подключаемых ресурсов

        /// <summary>
        /// </summary>
        public const int CORE_LEVEL = 0;

        /// <summary>
        /// </summary>
        public const int AUX_CORE_LEVEL = 10;

        /// <summary>
        /// 
        /// </summary>
        public const int RENDERER_LEVEL = 20;

        /// <summary>
        /// </summary>
        public const int EXTENSION_LEVEL = 100;

        #endregion

        #region Static members

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public static bool IsScriptManagerRequest
        {
            get
            {
#if NETCOREAPP
                return false;
#else
                var context = System.Web.HttpContext.Current;
                if (context == null)
                    return false;
                var res = HttpRequestHeaderHelper.HasQueryStringParameter(context.Request, "_TSM_HiddenField_");
                return res;
#endif
            }
        }

        /// <summary>
        ///   Возвращает текущий объект менеджера. Если надо - добавляет на страницу
        /// </summary>
        /// <param name = "control"></param>
        /// <returns></returns>
        public static PlatformScriptManager GetCurrent(Control control)
        {
#if !NETCOREAPP
            return GetCurrent(control.Page);
#else
            var view = control.FindNearestParentWebView() as Control;
            var manager = view.Controls.FirstOrDefault(c => c is PlatformScriptManager) as PlatformScriptManager;
            if (manager == null)
            {
                manager = new PlatformScriptManager {ID = "psm"};
                view.Controls.Add(manager);
            }

            return manager;
#endif
        }

#if !NETCOREAPP
        /// <summary>
        ///   Возвращает текущий объект менеджера. Если надо - добавляет на страницу
        /// </summary>
        /// <param name = "page"></param>
        /// <returns></returns>
        public new static PlatformScriptManager GetCurrent(Page page)
        {
            if (page == null)
                throw new ArgumentNullException("page");

            var manager = (PlatformScriptManager) page.Items[typeof(PlatformScriptManager)];
            if (manager == null)
            {
                manager = new PlatformScriptManager {ID = "psm"};
                page.Form.Controls.Add(manager);
            }

            return manager;
        }
#endif
        
        #endregion

        #region Public methods

        /// <summary>
        /// Регистрирует стартап-литерал
        /// </summary>
        /// <param name = "container"></param>
        /// <param name = "script"></param>
        public ScriptLiteral RegisterScriptLiteral(Control container, string script)
        {
            var scriptLiteral = new ScriptLiteral {Text = script, NeedTags = true};
            container.Controls.Add(scriptLiteral);
            return scriptLiteral;
        }

        /// <summary>
        /// Регистрирует стартап-литерал
        /// </summary>
        /// <param name = "container"></param>
        /// <param name="objectName"> </param>
        /// <param name="jso"> </param>
        public ScriptLiteral RegisterJavaScriptObject(Control container, string objectName, JavaScriptObject jso)
        {
            var sb = new StringBuilder();
            sb.AppendFormat("var {0} = ", objectName);
            sb.Append(jso.Value);
            var scriptLiteral = new ScriptLiteral {Text = sb.ToString(), NeedTags = true};
            container.Controls.Add(scriptLiteral);
            return scriptLiteral;
        }

        /// <summary>
        /// Регистрирует стартап-литерал
        /// </summary>
        /// <param name = "container"></param>
        /// <param name="id">Идентификатор литерала, если такой уже есть, то меняет его значение</param>
        /// <param name = "script"></param>
        public ScriptLiteral RegisterScriptLiteral(Control container, string id, string script)
        {
            Control c = container.FindControl(id);
            ScriptLiteral scriptLiteral;
            if (c == null)
            {
                scriptLiteral = new ScriptLiteral {Text = script, NeedTags = true};
                container.Controls.Add(scriptLiteral);
            }
            else
            {
                scriptLiteral = (ScriptLiteral) c;
                scriptLiteral.Text = script;
                scriptLiteral.NeedTags = true;
            }

            return scriptLiteral;
        }

        /// <summary>
        ///   Регистрирует подключаемый скрипт
        /// </summary>
        /// <param name = "key"></param>
        /// <param name = "scriptUrl"></param>
        public void RegisterScriptInclude(string key, string scriptUrl)
        {
            RegisterScriptInclude(key, scriptUrl, EXTENSION_LEVEL);
        }

        /// <summary>
        ///   Регистрирует подключаемый скрипт
        /// </summary>
        /// <param name = "key"></param>
        /// <param name = "scriptUrl"></param>
        /// <param name = "order">Порядок регистрации</param>
        public void RegisterScriptInclude(string key, string scriptUrl, int order)
        {
            m_scriptIncludes[key] = new ScriptIncludeReference {Name = scriptUrl, Order = order};
            if (!m_includeLevels.Contains(order))
                m_includeLevels.Add(order);
        }

        /// <summary>
        ///   Регистрирует подключаемый из ресурсов скрипт
        /// </summary>
        /// <param name = "key"></param>
        /// <param name = "type"></param>
        /// <param name = "name"></param>
        public void RegisterResourceScriptInclude(string key, Type type, string name)
        {
            RegisterResourceScriptInclude(key, type, name, EXTENSION_LEVEL);
        }

        /// <summary>
        ///   Регистрирует подключаемый из ресурсов скрипт
        /// </summary>
        /// <param name = "key"></param>
        /// <param name = "type"></param>
        /// <param name = "name"></param>
        /// <param name = "order">Порядок регистрации</param>
        public void RegisterResourceScriptInclude(string key, Type type, string name, int order)
        {
            m_scriptIncludes[key] = new ScriptIncludeReference {Type = type, Name = name, Order = order};
            if (!m_includeLevels.Contains(order))
                m_includeLevels.Add(order);
        }

        #endregion

        #region Framework overridables

#if !NETCOREAPP
        /// <summary>
        /// </summary>
        /// <param name = "e"></param>
        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);

            OnInitAsync(e).GetAwaiter().GetResult();
        }
#endif

#if NETCOREAPP
        protected override Task OnInitAsync(EventArgs eventArgs)
        {
            base.OnInitAsync(eventArgs);
#else
        protected virtual Task OnInitAsync(EventArgs eventArgs)
        {
            Page.Items[typeof(PlatformScriptManager)] = this;
            Page.PreRender += OnPagePrerender;
#endif
            return Task.CompletedTask;
        }

#endregion

        #region Private methods

#if !NETCOREAPP
        /// <summary>
        /// </summary>
        /// <param name = "sender"></param>
        /// <param name = "e"></param>
        private void OnPagePrerender(object sender, EventArgs e)
        {
            string version = '-' + ApplicationContext.Current.AppAssemblyVersion.ToString();

            m_includeLevels.Sort((obj1, obj2) => obj1 - obj2);
            foreach (int includeLevel in m_includeLevels)
            {
                foreach (KeyValuePair<string, ScriptIncludeReference> include in m_scriptIncludes)
                {
                    if (include.Value.Order != includeLevel)
                        continue;

                    if (include.Value.Type == null)
                    {
                        string path = include.Value.Name;
                        string extension = Path.GetExtension(path);

                        if (!ApplicationContext.Current.IsDebugBuild
                            && !Platform.IsUnix
                            && WebApplication.Current.EnableUrlRewriter
                            && String.CompareOrdinal(extension, ".js") == 0)
                        {
                            // ссылки на яваскрипты снабжаем номером версии для 
                            // борьбы с браузерными кешами в момент обновления приложения
                            string directoryPath = Path.GetDirectoryName(path) ?? string.Empty;
                            string versionedName = Path.GetFileNameWithoutExtension(path) + version;
                            string versionedPath = Path.Combine(directoryPath, versionedName + ".xjs")
                                .Replace('\\', '/');
                            RegisterClientScriptInclude(Page, typeof(PlatformScriptManager), include.Key,
                                versionedPath);
                        }
                        else
                            RegisterClientScriptInclude(Page, typeof(PlatformScriptManager), include.Key, path);
                    }
                    else
                        RegisterClientScriptResource(Page, include.Value.Type, include.Value.Name);
                }
            }
        }
#endif

        #endregion
    }
}