﻿Ext.define('Core.DbManager', {

    alias: 'Core.DbManager',

    extend: 'Ext.panel.Panel',
    
    listeners: {
        beforeclose: function () {
            this.RemoveKeys();
        }
    },

    constructor: function (args) {
        var me = this;

        me.response = args.res;

        me.callParent(arguments);

        me.InitServerResponseParams();

        me.CreateObjX();

        me.Refresh = me.Refresh.bind(me);

        me.isDbManager = true;
    },

    GetActiveTasks: function () {
        var index = this.tree.getStore().find('inProgress', true);

        return index !== -1;
    },

    Refresh: function () {
        var me = this,
            refreshTree = function () {
                AjaxRequest({
                    url: 'DbList/LoadOutcomeByKey',
                    mask: {
                        text: 'Обновление',
                        id: me
                    },
                    params: {
                        key: me.sessionKey
                    },
                    success: function (res) {
                        me.response = res;
                        me.InitServerResponseParams();
                    }
                });
            },
            activeTasks = me.GetActiveTasks();
        
        Dashboard.AddNavTab(me, true);

        if (activeTasks) {
            Ext.Msg.show({
                title: 'Подтверждение',
                message: 'Имеются активные операции<br><br>Обновить список баз данных?<br><br>Информация о завершении может быть утеряна',
                buttons: Ext.Msg.YESNO,
                icon: Ext.Msg.QUESTION,
                fn: function (btn) {
                    btn === 'yes' && refreshTree();
                }
            });
        } else {
            refreshTree();
        }
    },

    RefreshBranch: function (node) {
        node.set({
            'branchIsLoaded': false,
            'inProgress': false
        });
        this.GetBranchData(node, Ext.emptyFn);
    },

    ShowTaskCompliteMsg: function (text) {
        Ext.toast({
            html: '<p>' + text + '</p>',
            width: 200,
            align: 'tr'
        });
    },

    CreateObjX: function () {
        var me = this;

        me.ObjX = new _ObjX({
            ObjectCode: 'DB_LIST_MANAGER',
            renderPanel: me
        });

        if (me.ObjX.container && me.ObjX.container.destroy) {
            me.ObjX.container.destroy();
            delete me.ObjX.container;
        }
    },

    GetActiveDbNode: function () {
        if (!this.destroyed && this.tree && !this.tree.destroyed) {
            return this.tree.getSelection()[0];
        }
    },

    GetNodeDbName: function (node) {
        if (node && node.get && node.get('INode')) {
            return node.get('INode').DbName;
        } 
    },

    IsLoadableBranch: function (node) {
        if (!node || !node.isModel && isNaN(node)) return;
        //node = model or int
        var type = node.get('INode').SqlAdminNodeType;

        return (type == 10002 || type == 10018) && !node.get('branchIsLoaded');
    },

    InitServerResponseParams: function () {
        var me = this,
            res = me.response;
        if (res) {
            Ext.apply(me, {
                sessionKey: res.key,
                key: res.key,
                tooltip: res.server + ' - ' + res.user,
                currentServer: res.server,
                currentUser: res.user
            });
     
            if (me.tree) {
                me.tree.destroy();
                me.removeAll();
            }
            
            me.tree = me.CreateDbListTree();
            me.add(me.tree);

            me.response = null;
        }
    },

    CreateDbListTree: function () {
        var me = this;

        var tree = Ext_tree(me.response.tree, null, 'toNull', true);

        tree.on({
            'beforedestroy': function () {
                me.SaveDbKeys();
            },
            'boxready': function () {
                me.MarkCurrentDB();
            },
            'beforeitemcontextmenu': function (view, node, item, index, e) {
                e.stopEvent();

                if (node.get('inProgress')) return;

                var menu = node.get('extMenu'),
                    callback = function () {
                        me.ShowContextMenu(node, e);
                    };

                if (!menu && node.get('INode').SqlAdminNodeType == 10002 && !node.get('branchIsLoaded')) {
                    me.GetBranchData(node, callback);
                } else {
                    callback();
                }

                return false;
            },
            'beforeitemexpand': function (node) {
                return me.GetBranchData(node);
            },
            'itemdblclick': function (tree, node) {
                me.NodeDbClickHandler(node);
            }
        });

        return tree;
    },

    RefreshDbListTree: function () {
        var me = this,
            tree = me.tree,
            treeRoot = {
                children: me.response.tree[0]
            };

        tree.setRoot(treeRoot);
    },

    CreateContextMenu: function () {
        return Ext.create('Ext.menu.Menu', {
            items: [],
            ignoreParentClicks: true,
            listeners: {
                show: function () {
                    var menu = this,
                        e = menu.showEvent || event;
                    if (e) {
                        var pos = e.getXY ? e.getXY() : [event.x, event.y],
                            elPos = menu.getPosition(),
                            onItsPlace = Ext.Array.equals(pos, elPos);
                        
                        !onItsPlace && menu.setPosition(pos);
                    }
                }
            }
        });
    },

    GetDbKey: function (node) {
        var dbKey;
        while (!dbKey && node && node.get) {
            dbKey = node.get('dbKey');
            node = node.parentNode;
        }
        return dbKey;
    },

    SetDataSource: function (node) {
        var me = this;

        if (!me.ObjX.DataSource) {
            me.ObjX.DataSource = '';
        }

        me.ObjX.DataSource = me.GetDbKey(node);
    },

    NodeDbClickHandler: function (node) {
        var me = this,
            iNode = node.get('INode');

        me.SetDataSource(node);

        if (iNode && iNode.PluginName) {
            
            if (iNode.PluginName === 'Meta.GroupAdmin') {
                iNode.PluginName += '.Init';
            }

            var objx = ObjX,
                cb = function () {
                    ExecStrFunc(iNode.PluginName);
                };
            ExecFunctionInObjX(this.ObjX, cb);

            SetObjX(objx);
        }
    },

    GetExtNodeMenu: function (node) {
        var me = this,
            items = node.get('Menu').Items;
        return items.map(function (item) {
            if (!item.Code) {
                return '-';
            }

            var taskName = item.TaskName,
                type = item.NodeType,
                p = {
                    text: item.Name,
                    hidden: taskName && (taskName.includes('SqlQueryManager') || taskName.includes('SqlQueryRun') || taskName.includes('DbObjects')),
                    listeners: {},
                    code: item.Code,
                    handler: function () {
                        if (taskName) {
                            var cb = function () {
                                me.ObjX.mmx = new _MMX();
                                ExecStrFunc(taskName, me.ObjX.mmx);
                            };
                            ExecFunctionInObjX(me.ObjX, cb, me);
                        }
                    }
                };

            if (type === 61 || type === 6001) {
                Ext.apply(p, {
                    xtype: 'menucheckitem',
                    checked: item.Checked,
                    listeners: {
                        beforecheckchange: function () {
                            this.fireEvent('click');
                            return false;
                        }
                    }
                });
            }

            return p;
        });
    },

    ShowContextMenu: function (node, e) {
        var me = this,
            menu = me.contextMenu;

        if (!menu) {
            me.contextMenu = menu = me.CreateContextMenu();
        }

        menu.showEvent = e;

        var menuItems = node.get('extMenu');

        if (!menuItems) {
            menuItems = this.GetExtNodeMenu(node);
            node.set('extMenu', menuItems);
            node.set('Menu', null);
        }

        if (menuItems && menuItems.length) {
            menu.removeAll();
            menu.add(menuItems);
            menu.showAt(e.getXY());
        }
    },

    GetBranchData: function (node, callback) {
        var me = this;

        if (node.get('inProgress')) {
            return false;
        }

        if (me.IsLoadableBranch(node)) {
            me.MaskNode(node, 'Анализ БД');

            AjaxRequest({
                url: 'DbList/GetBranchData',
                params: {
                    key: me.sessionKey,
                    node: node.getData(),
                    dbKey: node.get('dbKey')
                },
                success: function (res) {
                    me.ApplyBranchToNode(node, res.data[0]);

                    node.set({
                        'branchIsLoaded': true,
                        'dbKey': res.dbKey
                    });

                    me.UnmaskNode(node);

                    callback ? callback() : node.expand();
                }
            });
            return false;
        }

    },

    MaskNode: function (node, text) {
        var string = Ext.String.format('{0} - <span style="color: blue;">{1}</span>', node.get('text'), text);
        node.set('text', string);
        node.set('inProgress', true);
    },

    UnmaskNode: function (node) {
        var string = node.get('text').split('-')[0].trim();
        node.set('text', string);
        node.set('inProgress', false);
    },

    ApplyBranchToNode: function (node, data) {
        var children = data.children,
            menu = data.Menu;

        node.removeAll();
        children && children.length && node.appendChild(children);
        node.set('Menu', menu);
        node.set('extMenu', null);
        node.commit();
    },

    MarkCurrentDB: function () {
        var me = this,
            tree = me.tree,
            record = tree.getStore().findRecord('text', WebProject.Info.DBName);

        record && tree.selectPath(record);
    },

    GetAllKeys: function () {
        var me = this,
            dbKeys = me.dbKeys || [];

        dbKeys.push(me.key);

        return dbKeys;
    },
    
    SaveDbKeys: function () {
        var me = this,
            store = me.tree.getStore();

        if (!me.dbKeys) me.dbKeys = [];

        if (store) {
            me.dbKeys = me.dbKeys.concat(store.collect('dbKey'));
        }
    },

    RemoveKeys: function (cb) {
        var keys = this.GetAllKeys();
        
        AjaxRequest({
            url: 'DbList/RemoveKeys',
            params: {
                keys: keys
            },
            success: function () {
                cb && cb.call && cb();
            }
        });
    },
    //db operations
    OnStartOperation: function (p) {
        if (!p) return;

        p.icon = 'ks-icon-database_manager48';
        p.isProcessStart = true;
        
        return Ext.create('Core.Bubble', p);
    },

    OnEndOperation: function (p) {
        //тут будет вызов метода добавления уведомления
        // title message
        if (!p) return;

        var btnText = p.btnText,
            btnClickCb = p.btnTextCb;

        if (p.protocol) {
            btnText = 'Показать протокол';
            btnClickCb = function () {
                ExtUtils.ShowText(p.protocol, {
                    title: p.title
                });
            };
        }

        Dashboard.NotificationManager.AddNotification({
            title: p.title,
            type: 'database',
            description: p.message,
            notSave: true,
            events: {
                buttonclick: btnClickCb
            },
            buttonConfig: {
                text: btnText
            }
        });
    },

    CheckNotificationsParams: function (p, defaultMsg) {
        if (!p) return null;

        var onStartParams = p.onStart,
            onEndParams = p.onEnd;

        if (onStartParams) {
            if (!onStartParams.title) {
                onStartParams.title = 'Запущена операция ' + onStartParams.str || defaultMsg || '';
            }
        }

        if (onEndParams) {
            if (!onEndParams.title) {
                onEndParams.title = 'Завершена операция ' + onEndParams.str || defaultMsg || '';
            }
        }

        return p;
    },

    ProxyCall: function (p) {
        var me = this,
            task = p.task,
            msg = p.msg,
            key = p.key,
            isLong = p.isLong,
            alertMsg = p.alertMsg,
            node = p.node,
            cb = p.callback,
            params = p.params || [],
            mask = p.mask,
            addParams = params.only ? params.args : params,
            runTask = function () {
                if (!params.only && node && node.getData) {
                    addParams = [{
                        stype: 'SqlAdminParam',
                        value: JSON.stringify(node.getData())
                    }].concat(addParams);
                }

                var proxyParams = {
                    key: me.sessionKey || key,
                    task: task,
                    param: addParams,
                    dbName: node ? node.get('INode').DbName || node.parentNode.get('INode').DbName : null
                };

                node && me.MaskNode(node, msg);

                me.CheckNotificationsParams(p, msg);

                var bubble = me.OnStartOperation(p.onStart);

                AjaxRequest({
                    url: 'DbList/ModelProxy',
                    mask: mask,
                    params: proxyParams,
                    success: function (res) {
                        node && !me.destroyed && me.UnmaskNode(node);

                        cb && cb.call(me, res);

                        bubble && !bubble.destroyed && bubble.close();

                        p.onEnd && me.OnEndOperation(p.onEnd);
                    },
                    error: function (res) {
                        node && !me.destroyed && me.UnmaskNode(node);

                        ShowErorr_Client(res);

                        bubble && !bubble.destroyed && bubble.close();

                        p.onEnd && me.OnEndOperation(p.onEnd);
                    }
                });
            };

        if (isLong || alertMsg) {
            Ext.Msg.show({
                title: 'Подтверждение',
                message: alertMsg || 'Выполнение операции может занять продолжительное время. Продолжить?',
                buttons: Ext.Msg.YESNO,
                icon: Ext.Msg.QUESTION,
                fn: function (btn) {
                    if (btn === 'yes') {
                        runTask();
                    }
                }
            });
        } else {
            runTask();
        }
    },

    CheckFreeSpace: function (p) {
        var task = p.task,
            callback = p.callback,
            node = p.node,
            params = p.params || [];

        var checkSpace = function (res) {
            if (!res) return ExtAlert('#err#Ошибка проверки свободного места на диске');

            if (res.Fail) {
                var checkSpaceMsg = 'Проверка свободного места для создания резервной копии.<br><br>' + (res.Message ? res.Message + '<br><br>' : '') + 'Все равно продолжить?';

                Ext.Msg.show({
                    title: 'Создание резервной копии базы данных',
                    message: checkSpaceMsg,
                    buttons: Ext.Msg.YESNO,
                    icon: Ext.Msg.QUESTION,
                    fn: function (btn) {
                        if (btn === 'yes') {
                            callback && callback();
                        }
                    }
                });
            } else {
                callback && callback();
            }
        };

        this.ProxyCall({
            task: task,
            node: node,
            callback: checkSpace,
            params: params
        });
    },

    GetDbSource: function (node, cb) {
        var me = this;

        me.ProxyCall({
            task: 'CheckSourceDatabase',
            node: node,
            callback: function (res) {
                var dbSource = res && res.Table && res.Table[0] && res.Table[0].fd ? res.Table[0].fd : '';
                if (dbSource) {
                    cb && cb(dbSource);
                } else {
                    ExtAlert('#err#Ошиба запроса расположения базы данных');
                }
            }
        });
    },

    ShowServerExplorer: function (textfield, selectFile) {
        if (!textfield) return;

        var me = this;

        Ext.create('Core.DbServerFolder', {
            autoShow: true,
            alwaysOnTop: true,
            modal: true,
            folderOnly: !selectFile,
            sessionKey: me.sessionKey,
            server: me.currentServer,
            user: me.currentUser,
            onSelectCallback: function (v) {
                textfield.setValue(v);
            }
        });
    },

    CreateCopyDbForm: function (node, path) {
        var me = this;
        return Ext.create('Ext.form.Panel', {
            isDict: true,
            border: 0,
            defaultType: 'textfield',
            defaults: {
                labelAlign: 'top',
                labelSeparator: '',
                anchor: '100%'
            },
            items: [{
                fieldLabel: 'Имя тестовой базы',
                name: 'dbNameNew',
                allowBlank: false,
                value: node.get('INode').DbName + '_test'
            },
            {
                fieldLabel: 'Путь для размещения файла резервной копии относительно сервера БД',
                name: 'tempPath',
                allowBlank: false,
                value: path,
                triggers: {
                    explorer: {
                        cls: 'ks-icon-folder_open',
                        handler: function () {
                            var textfield = this;
                            me.ShowServerExplorer(textfield);
                        }
                    }
                }
            },
            {
                fieldLabel: 'Путь для размещения файлов БД относительно сервера БД',
                name: 'dbPath',
                allowBlank: false,
                value: path,
                triggers: {
                    explorer: {
                        cls: 'ks-icon-folder_open',
                        handler: function () {
                            var textfield = this;
                            me.ShowServerExplorer(textfield);
                        }
                    }
                }
            },
            {
                xtype: 'checkbox',
                boxLabel: 'Удалить промежуточный файл резервной копии',
                name: 'delTemp',
                value: true
            }
            ]
        });
    },

    CreateBackupDbForm: function (backupFileName, path) {
        return Ext.create('Ext.form.Panel', {
            frame: true,
            isDict: true,
            defaultType: 'textfield',
            defaults: {
                labelAlign: 'top',
                labelSeparator: '',
                anchor: '100%'
            },
            items: [
                {
                    fieldLabel: 'Путь для размещения файла резервной копии относительно сервера БД',
                    name: 'backupPath',
                    allowBlank: false,
                    value: path,
                    triggers: {
                        explorer: {
                            cls: 'ks-icon-folder_open',
                            handler: function () {
                                var textfield = this;
                                me.ShowServerExplorer(textfield);
                            }
                        }
                    }
                },
                {
                    fieldLabel: 'Имя файла резервной копии',
                    name: 'backupFileName',
                    allowBlank: false,
                    value: backupFileName
                }
            ]
        });
    },

    CreateRestoreForm: function () {
        var me = this,

            getBackupFileInfo = function (selected) {
                var path = selected.get('physical_device_name');

                if (path.includes('Cannot open backup')) return;

                AjaxRequest({
                    url: 'DbList/GetBackupInfo',
                    mask: {
                        text: 'Получение информации о файле',
                        id: panel.up('window')
                    },
                    params: {
                        key: me.sessionKey,
                        path: path
                    },
                    success: setOutcome
                });
            },

            setOutcome = function (res) {
                var vm = panel.up('window').getViewModel(),
                    store = vm.get('backupFileInfo'),
                    source = vm.get('selectedFileInfo'),
                    fileInfoTable,
                    fileInfodata,
                    backupInfoTable,
                    backupInfoData;
    
                if (!res || res.Fail || !res.backupFileInfo || !res.backupInfo) {
                    var msg = 'Ошибка получения информации о резервной копии из файла. <br/ > Возможно, файл был удалён, либо сформирован на другой версии СУБД.<br/><br/>' + res.Message || '';

                    vm.set('fail', true);

                    ExtAlert('#err#' + msg);
                } else {
                    vm.set('fail', false);
                }

                vm.set('restoreFiles', res.backupFileInfo || null);

                fileInfoTable = _.get(res, 'backupFileInfo.Data.Tables[0]', null);
                fileInfodata = fileInfoTable ? DataUtils.GetDataFromTable(fileInfoTable) : [];
                backupInfoTable = _.get(res, 'backupInfo.Data.Tables[0]', null);
                backupInfoData = backupInfoTable ? DataUtils.GetDataFromTable(backupInfoTable)[0] : null;

                store.setData(fileInfodata);

                for (var key in source) {
                    vm.set('selectedFileInfo.' + key, backupInfoData ? backupInfoData[key] : '');
                }
            },

            panel = Ext.create('Ext.form.Panel', {
                layout: 'vbox',
                defaultType: 'textfield',
                defaults: {
                    labelAlign: 'top',
                    labelSeparator: '',
                    anchor: '100%',
                    width: '100%'
                },
                bodyPadding: 4,
                border: 0,
                items: [
                    {
                        fieldLabel: 'Путь и имя файла резервной копии относительно сервера БД',
                        name: 'path',
                        allowBlank: false,
                        editable: false,
                        triggers: {
                            explorer: {
                                cls: 'ks-icon-folder_open',
                                handler: function () {
                                    var textfield = this;
                                    me.ShowServerExplorer(textfield, true);
                                }
                            }
                        },
                        bind: {
                            value: '{filePath}'
                        }
                    },
                    {
                        fieldLabel: 'Восстановить базу данных как',
                        name: 'dbName',
                        allowBlank: false,
                        bind: {
                            value: '{dbName}'
                        }
                    },
                    {
                        xtype: 'checkbox',
                        boxLabel: 'Восстановить пользователей',
                        name: 'usrRestore',
                        bind: {
                            value: '{usrRestore}'
                        }
                    },
                    {
                        xtype: 'grid',
                        height: 250,
                        scrollable: 'y',
                        title: 'Информация о последних резервных копиях',
                        header: true,
                        margin: 0,
                        padding: 0,
                        selModel: {
                            selType: 'rowmodel',
                            mode: 'SINGLE'
                        },
                        listeners: {
                            selectionchange: function (grid, selected) {
                                selected && selected[0] && getBackupFileInfo(selected[0]);
                            }
                        },
                        bind: {
                            selection: '{filesInfoGridSelection}',
                            store: '{filesInfoStore}',
                            disabled: '{!filesInfoStore.count}'
                        },
                        columns: [
                            {
                                text: 'Физическое имя файла',
                                dataIndex: 'physical_device_name',
                                flex: 3
                            },
                            {
                                text: 'Размер (Б)',
                                xtype: 'numbercolumn',
                                format: '0,000',
                                dataIndex: 'backup_size',
                                align: 'right',
                                flex: 1
                            },
                            {
                                text: 'Начальное время',
                                dataIndex: 'backup_start_date',
                                flex: 1
                            },
                            {
                                text: 'Конечное время',
                                dataIndex: 'backup_finish_date',
                                flex: 1
                            }
                        ]
                    },
                    {
                        xtype: 'whitetabpanel',
                        flex: 1,
                        height: 300,
                        bind: {
                            disabled: '{fail}'
                        },
                        items: [
                            {
                                xtype: 'grid',
                                header: true,
                                title: 'Содержание файла',
                                store: null,
                                selModel: 'cellmodel',
                                plugins: {
                                    ptype: 'cellediting',
                                    clicksToEdit: 1
                                },
                                bind: {
                                    store: '{backupFileInfo}',
                                    disabled: '{fail}'
                                },
                                columns: [
                                    {
                                        text: 'Логическое имя',
                                        dataIndex: 'logicalname',
                                        flex: 1,
                                        readOnly: true
                                    },
                                    {
                                        text: 'Физическое имя',
                                        dataIndex: 'physicalname',
                                        flex: 3,
                                        editor: {
                                            completeOnEnter: false,
                                            field: {
                                                xtype: 'textfield',
                                                allowBlank: false
                                            }
                                        }
                                    }
                                ]
                            }, 
                            {
                                xtype: 'propertygrid',
                                title: 'Информация',
                                scrollable: 'y',
                                disableSelection: true,
                                listeners: {
                                    beforecellclick: function () {
                                        return false;
                                    }
                                },
                                sourceConfig: {
                                    collation: {
                                        displayName: 'Сличение (Сollation)'
                                    },
                                    databasecreationdate: {
                                        displayName: 'Дата создания',
                                        renderer: function (v) {
                                            var date = new Date(v);
                                            return date && date.getDate() ? date.toLocaleString() : '';
                                        },
                                        editor: false
                                    },
                                    recoverymodel: {
                                        displayName: 'Модель восстановления'
                                    },
                                    servername: {
                                        displayName: 'Сервер'
                                    },
                                    backupsize: {
                                        displayName: 'Размер',
                                        renderer: function (v) {
                                            return v ? Ext.util.Format.fileSize(v) : '';
                                        }
                                    }
                                },
                                bind: {
                                    disabled: '{fail}',
                                    source: {
                                        collation: '{selectedFileInfo.collation}',
                                        databasecreationdate: '{selectedFileInfo.databasecreationdate}',
                                        recoverymodel: '{selectedFileInfo.recoverymodel}',
                                        servername: '{selectedFileInfo.servername}',
                                        backupsize: '{selectedFileInfo.backupsize}'
                                    }
                                }
                            }
                        ]
                    }
                ]
            });

        return panel;
    },

    CreateNewMessageServerForm: function () {
        var me = this;
        return Ext.create('Ext.form.Panel', {
            border: 0,
            defaultType: 'textfield',
            width: 600,
            bodyPadding: 4,
            defaults: {
                padding: 10
            },
            items: [
                {
                    xtype: 'panel',
                    layout: 'card',
                    border: 0,
                    padding: 0,
                    margin: 0,
                    bind: {
                        activeItem: '{activeItemCard}'
                    },
                    items: [
                        {
                            xtype: 'fieldset',
                            title: 'Выбор базы данных:',
                            defaultType: 'textfield',
                            defaults: {
                                width: '100%'
                            },
                            items: [{
                                xtype: 'radiogroup',
                                layout: 'vbox',
                                bind: {
                                    value: '{db}'
                                },
                                items: [{
                                    boxLabel: 'Текущая база данных',
                                    name: 'selected',
                                    inputValue: 'current'
                                },
                                {
                                    boxLabel: 'Отдельная БД (рекомендуется)',
                                    name: 'selected',
                                    inputValue: 'other'
                                }
                                ]
                            },
                            {
                                fieldLabel: 'Имя базы',
                                labelAlign: 'top',
                                bind: {
                                    value: '{dbName}',
                                    disabled: '{isCurrentDb}'
                                }
                            },
                            {
                                fieldLabel: 'Каталог размещения',
                                labelAlign: 'top',
                                bind: {
                                    value: '{dbPath}',
                                    disabled: '{isCurrentDb}'
                                },
                                triggers: {
                                    explorer: {
                                        cls: 'ks-icon-folder_open',
                                        handler: function () {
                                            var textfield = this;
                                            me.ShowServerExplorer(textfield);
                                        }
                                    }
                                }
                            }
                            ]
                        },
                        {
                            xtype: 'fieldset',
                            
                            items: {
                                xtype: 'textfield',
                                editable: false,
                                fieldLabel: 'Наименование базы сервера сообщений',
                                labelAlign: 'top',
                                bind: {
                                    value: '{dbName}'
                                }
                            }
                        }
                    ]
                },
                {
                    xtype: 'fieldset',
                    title: 'Настройка сервера сообщений:',
                    layout: {
                        type: 'vbox',
                        align: 'stretch'
                    },
                    items: [
                        {
                            xtype: 'fieldcontainer',
                            layout: {
                                type: 'hbox',
                                align: 'center'
                            },
                            items: [
                                {
                                    xtype: 'label',
                                    flex: 1,
                                    maxWidth: 250,
                                    text: 'Интервал опроса агента клиента, сек',
                                    forId: 'queryInterval',
                                    margin: '0 8 0 0'
                                },
                                {
                                    xtype: 'ksspinnerfield',
                                    inputId: 'queryInterval',
                                    maxWidth: 70,
                                    flex: 1,
                                    minValue: 60,
                                    maxValue: 180 * 60,
                                    stepValue: 10,
                                    cycle: true,
                                    bind: {
                                        value: '{queryInterval}'
                                    }
                                }
                            ]
                        },
                        {
                            xtype: 'fieldcontainer',
                            layout: {
                                type: 'hbox',
                                align: 'center'
                            },
                            items: [
                                {
                                    xtype: 'label',
                                    flex: 1,
                                    maxWidth: 250,
                                    text: 'Время «жизни» сообщений, дней',
                                    forId: 'lifetime',
                                    margin: '0 8 0 0'
                                },
                                {
                                    xtype: 'ksspinnerfield',
                                    inputId: 'lifetime',
                                    maxWidth: 70,
                                    flex: 1,
                                    minValue: 1,
                                    maxValue: 365,
                                    stepValue: 1,
                                    bind: {
                                        value: '{lifetime}'
                                    }
                                }
                            ]
                        }
                    ]
                },
                {
                    xtype: 'displayfield',
                    value: '* Режим доступен только в версии «Про»'
                }
            ]
        });
    },

    CreateNewPrimaryDocumentationServerForm: function () {
        var me = this;

        return Ext.create('Ext.form.Panel', {
            border: 0,
            defaultType: 'textfield',
            width: 600,
            bodyPadding: 4,
            defaults: {
                padding: 10
            },
            items: [
                {
                    xtype: 'fieldset',
                    title: 'Выбор базы данных:',
                    defaultType: 'textfield',
                    defaults: {
                        width: '100%'
                    },
                    items: [
                        {
                            xtype: 'radiogroup',
                            layout: 'vbox',
                            bind: {
                                value: '{db}'
                            },
                            items: [
                                {
                                    boxLabel: 'Текущая база данных',
                                    name: 'selected',
                                    inputValue: 'current'
                                },
                                {
                                    boxLabel: 'Отдельная БД ',
                                    name: 'selected',
                                    inputValue: 'other'
                                }
                            ]
                        }
                    ]
                },
                {
                    xtype: 'fieldset',
                    title: 'Параметры:',
                    defaultType: 'textfield',
                    defaults: {
                        labelWidth: 140,
                    },
                    layout: {
                        type: 'vbox',
                        align: 'stretch'
                    },
                    bind: {
                        disabled: '{isCurrentDb}'
                    },
                    items: [
                        {
                            fieldLabel: 'Имя сервера',
                            disabled: true,
                            bind: {
                                value: '{serverName}'
                            }
                        },
                        {
                            fieldLabel: 'Имя БД',
                            bind: {
                                value: '{dbName}'
                            }
                        },
                        {
                            fieldLabel: 'Пользователь удаленного доступа',
                            disabled: true,
                            bind: {
                                value: '{removeUsername}'
                            }
                        },
                        {
                            fieldLabel: 'Пароль удаленного пользователя',
                            disabled: true,
                            bind: {
                                value: '{removePassword}'
                            }
                        },
                        {
                            fieldLabel: 'Каталог размещения',
                            triggers: {
                                explorer: {
                                    cls: 'ks-icon-folder_open',
                                    handler: function () {
                                        var textfield = this;
                                        me.ShowServerExplorer(textfield);
                                    }
                                }
                            },
                            bind: {
                                value: '{path}'
                            }
                        }
                    ]
                }
            ]
        });
    },

    CreateMailProfileEditForm: function () {
        return Ext.create('Ext.form.Panel', {
            border: 0,
            bodyPadding: 4,
            items: {
                title: 'Параметры профиля:',
                bodyPadding: 4,
                margin: 0,
                xtype: 'fieldset',
                defaultType: 'textfield',
                layout: {
                    type: 'vbox',
                    align: 'stretch'
                },
                defaults: {
                    labelWidth: 100,
                },
                items: [{
                    fieldLabel: 'Профиль:',
                    emptyText: 'Имя профиля',
                    bind: {
                        value: '{Profile}',
                        editable: '{!isEdit}'
                    }
                },
                {
                    fieldLabel: 'Описание:',
                    emptyText: 'Описание профиля',
                    bind: {
                        value: '{Description}'
                    }
                },
                {
                    fieldLabel: 'Аккаунт:',
                    emptyText: 'Почтовый аккаунт',
                    bind: {
                        value: '{Account}',
                        editable: '{!isEdit}'
                    }
                },
                {
                    fieldLabel: 'E-mail:',
                    emptyText: 'Почтовый ящик отправки',
                    bind: {
                        value: '{Email}'
                    }
                },
                {
                    fieldLabel: 'Отправитель:',
                    emptyText: 'Наименование отправителя',
                    bind: {
                        value: '{Sender}'
                    }
                },
                {
                    fieldLabel: 'Тема:',
                    emptyText: 'Тема сообщений',
                    bind: {
                        value: '{Subject}'
                    }
                },
                {
                    xtype: 'fieldcontainer',
                    layout: {
                        type: 'hbox',
                        align: 'center'
                    },
                    items: [{
                        xtype: 'textfield',
                        fieldLabel: 'Сервер:',
                        emptyText: 'Адрес сервера',
                        margin: '0 8 0 0',
                        flex: 2,
                        bind: {
                            value: '{Server}'
                        }
                    },
                    {
                        fieldLabel: 'Порт:',
                        xtype: 'ksspinnerfield',
                        labelWidth: false,
                        labelStyle: 'width: auto',
                        flex: 1,
                        minValue: 1,
                        maxValue: 65535,
                        stepValue: 1,
                        bind: {
                            value: '{Port}'
                        }
                    }
                    ]
                },
                {
                    xtype: 'fieldcontainer',
                    layout: {
                        type: 'hbox',
                        align: 'center'
                    },
                    defaultType: 'textfield',
                    items: [{
                        xtype: 'textfield',
                        fieldLabel: 'SMTP пользователь:',
                        flex: 1,
                        margin: '0 8 0 0',
                        bind: {
                            value: '{User}'
                        }
                    },
                    {
                        xtype: 'textfield',
                        inputType: 'password',
                        fieldLabel: 'SMTP пароль:',
                        labelWidth: false,
                        labelStyle: 'width: auto',
                        flex: 1,
                        bind: {
                            value: '{Password}'
                        }
                    }
                    ]
                }
                ]
            }
        });
    },

    CreateFormButtons: function (p) {
        p = p || {};
        var closeWin = function () {
                this.up('window') ? this.up('window').destroy() : null;
            },
            defaultHandler = function () {
                var window = this.up('window'),
                    form = window.down('form'),
                    cb = window.callback;

                if (!window || !form) return;

                if (form.isValid()) {
                    cb && cb.call && cb();
                    closeWin.call(this);
                } else {
                    ExtAlert('#msg#Заполните обязательные поля');
                }
            };

        return [
            {
                text: p.createBtnText || 'Cоздать',
                bind: p.createBtnBind || null,
                handler: p.createBtnHandler || defaultHandler
            }, {
                text: 'Отмена',
                handler: closeWin
            }
        ];
    },

    CheckDbConnection: function (node, callback) {
        var me = this;
        me.MaskNode(node, 'Проверка подключений');
        AjaxRequest({
            url: 'DbList/CheckDbConnection',
            params: {
                key: me.sessionKey,
                dbName: me.GetNodeDbName(node)
            },
            success: function (res) {
                me.UnmaskNode(node);
                if (res) {
                    if (res.Fail) {
                        ExtAlert('#err#Операция невозможна. БД используется другими пользователями.');
                    } else {
                        callback && callback();
                    }
                } else {
                    ExtAlert('#err#Операция невозможна.');
                }

            }
        });
    },

    StartSimpleDbTask: function (taskName, msg, cb) {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#err#Выберите БД');

        var dbName = me.GetNodeDbName(node),
            str = '«' + msg + ' БД ' + (dbName || '') + '»',
            title = 'Запустить процедуру ' + str + '?';

        me.ProxyCall({
            task: taskName,
            msg: msg,
            alertMsg: title,
            onStart: {
                str: str
            },
            onEnd: cb ? null : {
                str: str
            },
            node: node,
            callback: cb ? cb.bind(me, node) : Ext.emptyFn
        });
    },  

    CopyDb: function () {
        var me = this,
            node = me.GetActiveDbNode();

        me.GetDbSource(node, function (sourcePath) {
            me.ShowCopyDbForm({
                node: node,
                sourcePath: sourcePath
            });
        });
    },

    ShowCopyDbForm: function (p) {
        if (!p || !p.node || !p.sourcePath) return;

        var me = this,
            node = p.node,
            sourcePath = p.sourcePath,
            form = me.CreateCopyDbForm(node, sourcePath),
            buttons = me.CreateFormButtons();

        ExtUtils.ShowWindow({
            title: 'Создание тестовой копии базы на основе выбранной',
            items: form,
            modal: true,
            width: 600,
            layout: {
                type: 'vbox',
                align: 'stretch'
            },
            padding: 4,
            buttons: buttons,
            callback: me.StartDbCopyTask.bind(me, {
                node: node,
                form: form
            })
        });
    },

    StartDbCopyTask: function (p) {
        if (!p || !p.node || !p.form) return;

        var me = this,
            form = p.form,
            node = p.node,
            values = form && form.getForm ? form.getForm().getFieldValues() : null;

        if (!values) return;

        var dbNameTest = values.dbNameNew,
            tempPath = values.tempPath,
            dbPath = values.dbPath,
            delTemp = values.delTemp,
            dbName = me.GetNodeDbName(node),
            fileBackupPath = Replace(Ext.String.format('{0}\\{1}_{2}_tempCopy.bak', tempPath, dbName, StrDateTimeBD(new Date()), '\\\\', '\\')),
            params = [dbNameTest, delTemp, null, dbPath, fileBackupPath],
            callback = function () {
                var str = '«Cоздание резервной копии базы данных' + dbName + '»';
                me.ProxyCall({
                    task: 'CopyDataBase',
                    msg: 'cоздание копии',
                    isLong: true,
                    node: node,
                    params: params,
                    onStart: {
                        title: str
                    },
                    callback: function () {
                        var isRequiredRefresh = !me.destroyed && !me.GetActiveTasks();

                        me.OnEndOperation({
                            title: str,
                            msg: isRequiredRefresh ? 'Необходимо обновить список баз данных' : null,
                            btnText: isRequiredRefresh ? 'Обновить список БД' : null,
                            btnClickCb: isRequiredRefresh ? function() {
                                isRequiredRefresh && me.Refresh();
                            } : null
                        });
                    }
                });
            };

        me.CheckFreeSpace({
            task: 'CheckCopyDatabaseSpace',
            callback: callback,
            node: node,
            params: params || []
        });
    },


    BackupDb: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#msg#Выберите БД'); 
        
        me.GetDbSource(node, function (sourcePath) {
            path = sourcePath;
            me.ShowBackupForm({
                node: node,
                sourcePath: sourcePath
            });
        });
    },

    ShowBackupForm: function (p) {
        if (!p || !p.node || !p.sourcePath) return;

        var me = this,
            node = p.node,
            sourcePath = p.sourcePath,
            dbName = me.GetNodeDbName(node),
            date = StrDateTimeBD(new Date()),
            time = Replace(new Date().toLocaleTimeString(), ':', ''),
            backupFileName = Ext.String.format('{0}_{1}_{2}.bak', dbName, date, time),
            form = me.CreateBackupDbForm(backupFileName, sourcePath),
            buttons = me.CreateFormButtons();

        ExtUtils.ShowWindow({
            title: 'Создание резервной копии базы данных',
            items: form,
            modal: true,
            width: 600,
            layout: {
                type: 'vbox',
                align: 'stretch'
            },
            padding: 4,
            buttons: buttons,
            callback: me.StartBackupDbTask.bind(me, {
                node: node,
                form: form
            })
        });
    },

    StartBackupDbTask: function (p) {
        if (!p || !p.node || !p.form) return;

        var me = this,
            form = p.form,
            node = p.node,
            values = form && form.getForm ? form.getForm().getFieldValues() : null;

        if (!values) return;

        var dbName = me.GetNodeDbName(node),
            bFile = values.backupFileName,
            bPath = values.backupPath,
            path = Replace(Ext.String.format('{0}\\{1}', bPath, bFile), '\\\\', '\\').replace(/[^\x20-\x7E]/g, ''),
            params = {
                only: true,
                args: [dbName, path, false]
            },
            callback = function () {
                var str = '«Cоздание резервной копии базы данных ' + dbName + '»';
                me.ProxyCall({
                    task: 'BackupDatabase',
                    msg: 'создание резервной копии',
                    node: node,
                    params: params,
                    onStart: {
                        title: str
                    },
                    onEnd: {
                        title: str
                    }
                });
            };

        me.CheckFreeSpace({
            task: 'CheckBackupSpace',
            node: node,
            callback: callback,
            params: params
        });
    },


    RestoreDb: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#msg#Выберите БД'); 

        me.MaskNode(node, 'Получение свойств БД');
        AjaxRequest({
            url: 'DbList/RestoreLoadOutcome',
            params: {
                key: me.sessionKey,
                dbName: me.GetNodeDbName(node)
            },
            success: function (res) {
                me.UnmaskNode(node);
                if (res && res.Data && res.Data.Tables) {
                    var backupList = DataUtils.GetDataFromTable(res.Data.Tables[0]);
                    me.ShowRestoreDbForm(node, backupList);
                } else {
                    ExtAlert('#err#Ошибка получения списка резервных копий БД');
                }
            }
        });
    },

    ShowRestoreDbForm: function (node, backupList) {
        var me = this,
            buttons = me.CreateFormButtons({
                createBtnBind: {
                    disabled: '{fail}'
                },
                createBtnText: 'Восстановить'
            }),
            form = me.CreateRestoreForm(),
            win = ExtUtils.ShowWindow({
                title: 'Восстановление резервной копии базы данных',
                items: form,
                modal: true,
                width: 'auto',
                layout: {
                    type: 'vbox',
                    align: 'stretch'
                },
                viewModel: {
                    stores: {
                        filesInfoStore: {
                            data: backupList
                        },
                        backupFileInfo: {
                            data: ''
                        }
                    },
                    formulas: {
                        filePath: {
                            get: function (get) {
                                var selected = get('filesInfoGridSelection');
                                if (selected) {
                                    var path = selected.get('physical_device_name');
                                    return path && path.includes && path.includes('Cannot open backup') ? '' : path;
                                }
                            }
                        }
                    },
                    data: {
                        filesInfoGridSelection: null,
                        filePath: null,
                        dbName:  me.GetNodeDbName(node),
                        usrRestore: false,
                        selectedFile: null,
                        fail: true,
                        selectedFileInfo: {
                            collation: '',
                            databasecreationdate: '',
                            recoverymodel: '',
                            servername: '',
                            backupsize: ''
                        }
                    }
                },
                padding: 4,
                buttons: buttons,
                callback: function () {
                    me.StartRestoreDbTask({
                        node: node,
                        vm: win.getViewModel()
                    });
                }
            });
    },

    StartRestoreDbTask: function (p) {
        if (!p || !p.node || !p.vm) return;

        var me = this,
            vm = p.vm,
            params = p.params || {},
            node = p.node;

        if (!params.checked && vm) {
            
            var path = vm.get('filePath'),
                dbName = vm.get('dbName'),
                usrRestore = vm.get('usrRestore'),
                store = vm.get('backupFileInfo'),
                restoreFiles = vm.get('restoreFiles'),
                outcomeIsOk = DataUtils.ConvertFromDataToOutcome(restoreFiles, [store], [1]);

            if (!outcomeIsOk) return;

            params = {
                key: me.sessionKey,
                dbName: dbName,
                path: path,
                usrRestore: usrRestore,
                restoreFiles: restoreFiles
            };
        }

        node && me.MaskNode(node, 'восстановление');

        AjaxRequest({
            url: 'DbList/RestoreDatabase',
            params: params,
            success: function (res) {
                node && me.UnmaskNode(node);
                me.CheckRestoreDbResult({
                    res: res,
                    params: params,
                    node: node
                });
            }
        });
    },

    CheckRestoreDbResult: function (p) {
        var me = this,
            res = p.res,
            params = p.params,
            message = [],
            msgParams = {
                title: 'Восстановление базы из резервной копии',
                icon: Ext.Msg.QUESTION,
                buttons: Ext.Msg.OK
            };

        if (res) {
            if (res.Fail) {
                if (res.Message) {
                    message.push(res.Message);
                }
                if (res.Code > 0) {
                    message.push('<br>Все равно продолжить?');
                    msgParams.buttons = Ext.Msg.YESNO;
                    msgParams.fn = function (btn) {
                        if (btn === 'yes') {
                            params.checks = res.Code;
                            me.StartRestoreDbTask(p);
                        }
                    };
                }
            } else {
                me.OnEndOperation({
                    title: 'Восстановление базы из резервной копии «' + dbName + '» завершено.'
                });
                return;
            }
        } else {
            message.push('Неверный ответ сервера!');
        }

        if (message.length) {
            msgParams.message = message.join('<br>');
            Ext.Msg.show(msgParams);
        }
    },


    DeleteDb: function () {
        var me = this,
            node = me.GetActiveDbNode(),
            startTask = function () {
                var str = '«Удаление БД' + dbName + '»';
                me.ProxyCall({
                    task: 'DeleteDatabase',
                    msg: 'Удаление',
                    alertMsg: 'Удалить базу данных «' + dbName + '»?',
                    node: node,
                    params: {
                        only: true,
                        args: [dbName, null]
                    },
                    onStart: {
                        title: str
                    },
                    onEnd: {
                        title: str
                    },
                    callback: function () {
                        node.remove();
                    }
                });
            };

        if (!node && !node.get) return ExtAlert('#msg#Выберите БД'); 

        me.CheckDbConnection(node, startTask);
    },


    RenameDb: function () {
        var me = this,
            node = me.GetActiveDbNode(),
            dbName = me.GetNodeDbName(node),
            startTask = function (newDbName) {
                me.ProxyCall({
                    task: 'RenameDatabase',
                    msg: 'переименование',
                    node: node,
                    params: {
                        only: true,
                        args: [dbName, newDbName]
                    },
                    callback: function () {
                        node.set('text', newDbName);
                        node.commit();
                    }
                });
            };

        if (!node && !node.get) return ExtAlert('#err#Выберите БД');

        me.CheckDbConnection(node, function () {
            Ext.Msg.prompt(
                'Переименовать базу данных',
                'Введите новое наименование',
                function (btn, newDbName) {
                    btn == 'ok' && newDbName && newDbName.split && newDbName.length && newDbName !== dbName && startTask(newDbName);
                }, me, false, dbName);
        });
    },


    UpdateStatistic: function () {
        this.StartSimpleDbTask('UpdateStatisticDatabase', 'обновление статистики');
    },


    IndexDefragmentation: function () {
        this.StartSimpleDbTask('IndexDefragmentation', 'дефрагментация');
    },


    ReIndex: function () {
        this.StartSimpleDbTask('ReindexDatabase', 'переиндексация');
    },


    ShrinkDatabase: function () {
        this.StartSimpleDbTask('ShrinkDatabase', 'сжатие');
    },


    CheckDatabase: function () {
        var me = this,
            callback = function (node, res) {
                me.OnEndOperation({
                    title: 'Проверка БД «' + me.GetNodeDbName(node) + '» завершена',
                    protocol: '<pre>' + res.Message + '</pre>'
                });
            };

        me.StartSimpleDbTask('CheckDatabase', 'проверка', callback);
    },


    DatabaseProperties: function () {
        var me = this,
            node = me.GetActiveDbNode(),
            dbName = me.GetNodeDbName(node),
            str = '«Анализ свойств БД ' + dbName + '»';

        me.ProxyCall({
            task: 'ReadDatabaseProperty',
            msg: 'запрос информации',
            node: node,
            params: {
                only: true,
                args: ['DataSource']
            },
            onStart: {
                str: str
            },
            callback: function (res) {
                me.OnEndOperation({
                    title: str,
                    protocol: res
                });
            }
        });
    },


    PerfInformation: function() {
        var me = this,
            node = me.GetActiveDbNode(),
            dbName = me.GetNodeDbName(node),
            str = '«Анализ производительности БД ' + dbName + '»';

        me.MaskNode(node, 'Анализ производительности');

        me.onStartParams({
            title: 'Запущена операция ' + str
        });

        AjaxRequest({
            url: 'DbList/GetPerformanceInformation',
            params: {
                key: me.sessionKey,
                dbName: dbName
            },
            success: function(res) {
                node.UnmaskNode(node);
                var isSuccess = res && res.data && res.fileName;
                me.OnEndOperation({
                    title: 'Операция ' + str + (isSuccess ? ' завершена.' : ' завершилась с ошибкой.'),
                    btnText: isSuccess ? 'Скачать результат анализа' : null,
                    btnClickCb: isSuccess ? function() {
                        FileUtils.SaveText(res.data, res.fileName);
                    } : null
                });
            }
        });
    },


    ProcessList: function () {
        var me = this,
            node = me.GetActiveDbNode(),
            dbName = me.GetNodeDbName(node),
            callback = function (res) {
                Dashboard.GotoTab({
                    ObjectCode: 'DbProcessList_' + me.currentServer + '.' + dbName,
                    dbNode: node,
                    title: 'Список процессов ' + me.currentServer + '.' + dbName,
                    CustomRender: me.RenderDbProcessList.bind(me, {
                        dataLoad: 'DbList/LoadDbProcessListOutcome',
                        refreshData: 'DbList/LoadDbProcessListOutcome',
                        doNotRefresh: true,
                        outcome: res,
                        supressTotal: true,
                        supressDetails: true,
                        supressSummary: true,
                        supressFilter: true,
                        compConfig: {
                            columns: DataUtils.GetGridColumnsFromTable(res.Data.Tables[0])
                        }
                    }),
                    oim: ''
                });
            };

        me.FetchDbProcessList(node, callback);
    },

    RenderDbProcessList: function (p) {
        var me = this,
            tbar = Ext.create('Ext.toolbar.Toolbar', {
                dock: 'top',
                items: [
                    {
                        iconCls: 'ks-icon-delete',
                        tooltip: 'Удалить процесс(ы)',
                        handler: function () {
                            var processes = list.grid.getSelection();
                            me.KillProcess.call(list, {
                                processes: processes,
                                sessionKey: me.sessionKey
                            });
                        }
                    },
                    '-', 
                    {
                        iconCls: 'ks-icon-refresh',
                        tooltip: 'Обновить',
                        handler: function () {
                            list.Refresh();
                        }
                    }
                ]
            }),
            objx = ObjX,
            list = objx.List = Ext.create('Core.List', Ext.apply(p, {
                callback: function () {
                    objx.grid.addDocked(tbar);
                }
            }));

        list.Refresh = function () {
            var cb = function (res) {
                list.outcome = res;
                list.BindStore();
            };
            me.FetchDbProcessList(null, cb);
        };
    },

    FetchDbProcessList: function (node, cb) {
        var me = this,
            mask = null;

        if (node) {
            me.MaskNode(node, 'загрузка списка процессов');
        } else {
            mask = {
                text: 'Обновление списка процессов',
                id: ObjX.containerMain
            };
        }

        AjaxRequest({
            url: 'DbList/LoadDbProcessListOutcome',
            mask: mask,
            params: {
                key: me.key,
                dbName: me.GetNodeDbName(node || ObjX.dbNode)
            },
            success: function (res) {
                node && me.UnmaskNode(node);
                cb && cb(res);
            }
        });
    },

    KillProcess: function (p) {
        if (!p || !p.processes || !p.sessionKey) return;

        var me = this,
            pids = p.processes.map(function (rec) {
                return rec.get('spid');
            });

        AjaxRequest({
            url: 'DbList/KillProcess',
            mask: {
                text: 'Удаление процесса(в)',
                id: me.ObjX.containerMain,
            },
            params: {
                key: p.sessionKey,
                pids: pids
            },
            success: function (res) {
                if (res && !res.Fail) {
                    me.Refresh();
                } else {
                    ExtAlert('#err#Ошибка удаления процессов.' + res && res.Message ? res.Message : '');
                }
            }
        });
    },


    CreateMessageServer: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#msg#Выберите почтовый сервер БД');

        
        if (node.get('INode').SqlAdminNodeType !== 10009) return;
        
        var dbName = me.GetNodeDbName(node.parentNode || null),
            form = me.CreateNewMessageServerForm(),
            buttons = me.CreateFormButtons(),
            win = ExtUtils.ShowWindow({
                title: 'Создание сервера сообщений',
                items: form,
                width: 450,
                buttons: buttons,
                callback: function () {
                    me.StartCreateMessageServerTask({
                        node: node,
                        vm: win.getViewModel()
                    });
                },
                viewModel: {
                    data: {
                        db: {
                            selected: 'other'
                        },
                        dbName: dbName + '_messaging',
                        dbPath: '',
                        queryInterval: 300,
                        lifetime: 180,
                        activeItemCard: 0
                    },
                    formulas: {
                        isCurrentDb: {
                            get: function (get) {
                                return get('db.selected') === 'current';
                            }
                        }
                    }
                }
            });
    },

    StartCreateMessageServerTask: function (p) {
        if (!p || !p.node || !p.vm) return;

        var me = this,
            node = p.node,
            vm = p.vm,
            dbName = vm.get('dbName'),
            dbPath = vm.get('dbPath'),
            queryInterval = vm.get('queryInterval'),
            lifetime = vm.get('lifetime'),
            str = '«Создание сервера сообщений для базы данных ' + dbName + '»';

        if (vm.get('db.selected') === 'current') {
            dbName = me.GetNodeDbName(node.parentNode);
            dbPath = null;
        } 

        me.ProxyCall({
            task: 'CreateMessageServer',
            msg: 'Создание сервера сообщений',
            params: [dbName, dbPath, queryInterval, lifetime],
            onStart: {
                str: str
            },
            callback: function (res) {
                var isRequiredRefresh = !me.destroyed && !me.GetActiveTasks();
                me.OnEndOperation({
                    title: 'Операция ' + str + (res && res.Fail ? ' завершилась с ошибкой.' : ' завершена.'),
                    message: isRequiredRefresh ? 'Необходимо обновить список баз данных' : null,
                    btnText: isRequiredRefresh ? 'Обновить список БД' : null,
                    btnClickCb: isRequiredRefresh ? function() {
                        isRequiredRefresh && me.Refresh();
                    } : null
                });
            }
        });

    },


    AttachMessageServer: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#msg#Выберите БД');

        var dbNode = node.parentNode;

        me.ProxyCall({
            task: 'SearchMessageServers',
            node: dbNode,
            mask: {
                text: 'Загрузка доступных серверов сообщений',
                id: me
            },
            callback: me.ShowMessageServersList.bind(me, dbNode)
        });
    },


    ShowMessageServersList: function (node, list) {
        list = list.map(function (name) {
            return {
                text: name,
                leaf: true
            };
        });

        var me = this,
            buttons = me.CreateFormButtons({
                createBtnText: 'Присоединить',
                createBtnHandler: function () {
                    var tree = this.up('window').down('treelist'),
                        selected = tree.getSelection();

                    if (!selected) return ExtAlert('#msg#Выберите сервер сообщений');

                    me.StartAttachMessageServerTask(node, selected.get('text'));
                    this.up('window').close();
                }
            }),
            win = ExtUtils.ShowWindow({
                title: 'Список доступных серверов сообщений',
                width: 480,
                height: 340,
                padding: 4,
                layout: 'fit',
                items: {
                    xtype: 'panel',
                    scrollable: 'y',
                    items: {
                        xtype: 'treelist',
                        store: {
                            root: {
                                expanded: true,
                                children: list
                            }
                        },
                        listeners: {
                            dblclick: {
                                element: 'element',
                                fn: function (e, el) {
                                    me.StartAttachMessageServerTask(node, el.innerText);
                                    win.close();
                                }
                            }
                        }
                    }
                },
                buttons: buttons
            });
    },

    StartAttachMessageServerTask: function (node, serverName) {
        var me = this,
            str = '«Подключение сервера сообщений ' + serverName + '»';

        me.ProxyCall({
            task: 'AttachMessageServer',
            node: node,
            msg: 'Подключение сервера сообщений',
            params: [serverName],
            onStart: {
                str: str
            },
            onEnd: {
                str: str
            },
            callback: function (res) {
                me.RefreshBranch(node);
            }
        });
    },


    DetachMessageServer: function () {
        var me = this,
            node = me.GetActiveDbNode();
        
        if (!node && !node.get) return ExtAlert('#msg#Выберите БД');

        var dbNode = node.parentNode,
            messageServerName = node.get('INode').LinkedServer,
            dbName = me.GetNodeDbName(dbNode);

        if (!messageServerName) return ExtAlert('#err#Отсутствует сервер сообщений');

        dbNode.get('INode').LinkedServer = messageServerName;

        var str = '«Отсоединение сервера сообщений для базы данных' + dbName + '»';
        
        me.ProxyCall({
            task: 'DetachMessageServer',
            node: dbNode,
            msg: 'Отсоединение сервера сообщений',
            params: [messageServerName],
            onStart: {
                str: str
            },
            onEnd: {
                str: str
            }
        });
    },


    ConfigMessageServer: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#msg#Выберите БД');

        node = node.parentNode;

        me.ProxyCall({
            task: 'LoadMessageServerConfig',
            node: node,
            msg: 'Загрузка свойство сервера сообщений',
            callback: me.ShowMessageServerConfigEditForm.bind(me, node)
        });
    },

    ShowMessageServerConfigEditForm: function (node, res) {
        var me = this,
            form = me.CreateNewMessageServerForm(),
            buttons = me.CreateFormButtons({
                createBtnText: 'Сохранить'
            }),
            win = ExtUtils.ShowWindow({
                title: 'Конфигурация сервера сообщений',
                items: form,
                width: 450,
                buttons: buttons,
                callback: function () {
                    me.StartMessageServerUpdateConfigTask({
                        node: node,
                        vm: win.getViewModel()
                    });
                },
                viewModel: {
                    data: {
                        activeItemCard: 1,
                        dbName: res.MessageServerName,
                        queryInterval: res.ClientPollingInterval,
                        lifetime: res.MessageLifeDuration
                    }
                }
            });
    },

    StartMessageServerUpdateConfigTask: function (p) {
        if (!p || !p.node || !p.vm) return;

        var me = this,
            node = p.node,
            vm = p.vm,
            dbName = me.GetNodeDbName(node),
            str = '«Обновление конфигурации сервера сообщений ' + dbName + '»';

        me.ProxyCall({
            task: 'UpdateMessageServerConfig',
            node: p.node,
            params: [+vm.get('queryInterval'), +vm.get('lifetime')],
            msg: 'Обновление конфигурации сервера сообщений',
            onStart: {
                str: str
            },
            onEnd: {
                str: str
            }
        });
    },


    CreatePrimaryDocumentServer: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node && !node.get) return ExtAlert('#msg#Выберите хранилище первичных документов');

        if (node.get('INode').SqlAdminNodeType !== 10010) return;

        var dbNode = node.parentNode,
            dbName = me.GetNodeDbName(node.parentNode || null),
            form = me.CreateNewPrimaryDocumentationServerForm(),
            buttons = me.CreateFormButtons(),
            win = ExtUtils.ShowWindow({
                title: 'Создание хранилища первичных документов',
                items: form,
                width: 450,
                buttons: buttons,
                callback: function () {
                    me.StartCreatePrimaryDocumentServerTask({
                        node: dbNode,
                        vm: win.getViewModel()
                    });
                },
                viewModel: {
                    data: {
                        db: {
                            selected: 'current'
                        },
                        dbName: dbName + '_primarydocs',
                        path: '',
                        serverName: '',
                        remoteUsername: '',
                        removePassword: ''
                    },
                    formulas: {
                        isCurrentDb: {
                            get: function (get) {
                                return get('db.selected') === 'current';
                            }
                        }
                    }
                }
            });
    },

    StartCreatePrimaryDocumentServerTask: function (p) {
        if (!p || !p.node || !p.vm) return;

        var me = this,
            node = p.node,
            vm = p.vm,
            dbName = vm.get('dbName'),
            dbPath = vm.get('path'),
            serverName = vm.get('serverName'),
            remoteUsername = vm.get('remoteUsername'),
            removePassword = vm.get('removePassword'),
            mode = +!vm.get('isCurrentDb');

        if (mode === 0 || node === 1) {
            serverName = null;
            remoteUsername = null;
            removePassword = null;
        }

        if (mode === 0) {
            dbName = me.GetNodeDbName(node);
        }

        var str = '«Создание хранилища первичных документов для базы данных ' + dbName + '»';

        me.ProxyCall({
            task: 'CreatePrimaryDocumentsServer',
            msg: 'Создание хранилища первичных документов',
            node: node,
            params: [
                mode,
                serverName,
                dbName,
                remoteUsername,
                removePassword,
                dbPath
            ],
            onStart: {
                str: str
            },
            callback: function (res) {
                me.OnEndOperation({
                    title: 'Операция ' + str + (res && res.Fail ? ' завершилась с ошибкой.' : ' завершена.'),
                });
                me.RefreshBranch(node);
            }
        });
    },


    DeletePrimaryDocumentServer: function () {
        var me = this,
            node = me.GetActiveDbNode();
        
        if (!node && !node.get) return ExtAlert('#msg#Выберите хранилище первичных документов');

        if (node.get('INode').SqlAdminNodeType !== 10010) return;

        var dbNode = node.parentNode,
            str = '«Удаление хранилища первичных документов для базы данных ' + me.GetNodeDbName(dbNode) + '»';

        me.ProxyCall({
            task: 'DeletePrimaryDocumentsServer',
            message: 'Удаление хранилища первичных документов',
            alertMsg: 'При удалении все первичные документы будут потеряны.<br>Если в качестве хранилища используется отдельная БД, то она также будет удалена.',
            node: dbNode,
            onStart: {
                str: str
            },
            callback: function (res) {
                me.OnEndOperation({
                    title: 'Операция ' + str + (res && res.Fail ? ' завершилась с ошибкой.' : ' завершена.')
                });
                me.RefreshBranch(dbNode);
            }
        });
    },


    CreateMailProfile: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node) return;

        if (node.get('INode').SqlAdminNodeType === 10019) {
            me.FetchMailProfileData(node.get('INode').LinkedServer, me.EditMailProfile.bind(me, node.parentNode));
        } else {
            me.EditMailProfile(node);
        }
        
    },

    FetchMailProfileData: function (linkedServer, cb) {
        if (!linkedServer) return ExtAlert('#err#Ошибка');

        var me = this,
            profile = linkedServer.split('|')[0];

        if (!profile) return ExtAlert('#err#Ошибка');

        AjaxRequest({
            url: 'DbList/GetMailProfileData',
            mask: {
                text: 'Загрузка данных почтового профиля',
                id: me
            },
            params: {
                key: me.sessionKey,
                profile: profile
            },
            success: function (res) {
                cb && cb.call && cb(res);
            }
        });
    },

    EditMailProfile: function (node, data) {
        var me = this,
            form = me.CreateMailProfileEditForm(),
            buttons = me.CreateFormButtons({
                createBtnText: data ? 'Сохранить' : null
            });
        
        if (data) {
            data.isEdit = true;
        }

        var win = ExtUtils.ShowWindow({
            title: (data ? 'Создание' : 'Редактирование') + ' почтового профиля',
            items: form,
            width: 500,
            buttons: buttons,
            callback: function () {
                me.StartCreateMailProfileTask({
                    node: node,
                    vm: win.getViewModel()
                });
            },
            viewModel: {
                data: data || {
                    ProfileId: 0,
                    AccountId: 0,
                    Profile: '',
                    Description: '',
                    Account: '',
                    Email: '',
                    Sender: '',
                    Subject: '',
                    Server: '',
                    Port: 25,
                    User: '',
                    Password: ''
                }
            }
        });
    },

    StartCreateMailProfileTask: function (p) {
        if (!p || !p.node || !p.vm) return;

        var me = this,
            node = p.node,
            vm = p.vm,
            params = vm.getData();

        AjaxRequest({
            url: 'DbList/CreateDbMailProfile',
            params: {
                key: me.sessionKey, 
                outcome: params
            },
            success: function (res) {
                if (res && res.Fail) ExtAlert('#err#Ошибка создания почтового профиля' + res.Message || '');
                me.RefreshBranch(node);
            }
        });

    },


    DeleteMailProfile: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node || node && (node.get('INode').SqlAdminNodeType !== 10019 || !node.get('INode').LinkedServer)) return;

        var linkedServer = node.get('INode').LinkedServer.split('|'),
            profile = linkedServer[0],
            account = linkedServer[1];

        if (!profile || !account) return ExtAlert('#err#Отсутствуют данные почтового профиля');

        AjaxRequest({
            url: 'DbList/RemoveMailProfile',
            mask: {
                text: 'Удаление почтового профиля',
                mask: me
            },
            params: {
                key: me.sessionKey,
                profile: profile,
                account: account
            },
            success: function (res) {
                if (res && res.Fail) ExtAlert('#err#Ошибка удаления почтового профиля' + res.Message || '');
                me.RefreshBranch(node.parentNode);
            }
        });
    },


    OpenMessageList: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node || node && node.get('INode').SqlAdminNodeType !== 10023) return;

        var dbNode = node.parentNode,
            dbName = me.GetNodeDbName(dbNode),
            callback = function (res) {
                Dashboard.GotoTab({
                    ObjectCode: 'DbMessagesList_' + me.currentServer + '.' + dbName,
                    dbNode: node,
                    title: 'Общий просмотр сообщений ' + me.currentServer + '.' + dbName,
                    CustomRender: me.RenderMessageList.bind(me, {
                        dataLoad: 'DbList/LoadDbProcessListOutcome',
                        refreshData: 'DbList/LoadDbProcessListOutcome',
                        doNotRefresh: true,
                        localData: res.Messages,
                        supressTotal: true,
                        supressDetails: true,
                        supressSummary: true,
                        supressFilter: true,
                        DoubleClick: function (grid, td, cellIndex, record) {
                            me.ShowMessage(record.get('ID'));
                        },
                        compConfig: {
                            columns: [{
                                width: 16,
                                dataIndex: 'State',
                                text: 'Статус',
                                xtype: 'templatecolumn', 
                                tpl: '<tpl if="State === 2"><div style="width:16px;height:16px;margin-left:auto;margin-right:auto;" class="ks-icon-message_important"></div></tpl>'
                            },
                            {
                                dataIndex: 'IsRead',
                                text: 'Прочитано',
                                xtype: 'templatecolumn',
                                tpl: new Ext.XTemplate([
                                    '<tpl if="IsRead"><div style="width:16px;height:16px;margin-left:auto;margin-right:auto;" class="ks-icon-message_read"></div>',
                                    '<tpl else><div style="width:16px;height:16px;margin-left:auto;margin-right:auto;" class="ks-icon-message"></tpl>'
                                ])
                            },
                            {
                                dataIndex: 'RelatedDocuments',
                                text: 'Связанные документы',
                                xtype: 'templatecolumn',
                                tpl: '<tpl if="RelatedDocuments && RelatedDocuments.length"><div style="width:16px;height:16px;margin-left:auto;margin-right:auto;" class="ks-icon-document"></div></tpl>'
                            },
                            {
                                dataIndex: 'AttachedFiles',
                                text: 'Прикрепленные сообщения',
                                xtype: 'templatecolumn',
                                tpl: '<tpl if="AttachedFiles && AttachedFiles.length"><div style="width:16px;height:16px;margin-left:auto;margin-right:auto;" class="ks-icon-message_attach"></div></tpl>'
                            },
                            {
                                dataIndex: 'Sender',
                                text: 'От',
                                renderer: function (v) {
                                    return v && v.Value && v.Value[0] ? v.Value[0] : '';
                                }
                            },
                            {
                                dataIndex: 'SendTo',
                                text: 'Кому',
                                renderer: function (v) {
                                    if (v) {
                                        var result = [];
                                        for (var key in v) {
                                            result.push(v[key][0]);
                                        }
                                        return result.join(', ');
                                    }
                                }
                            },
                            {
                                dataIndex: 'Date',
                                text: 'Дата отправки',
                                xtype: 'datecolumn',
                                format: 'd M Y h:i:s'
                            },
                            {
                                dataIndex: 'Theme',
                                text: 'Тема'
                            },
                            {
                                dataIndex: 'NotReadUsers',
                                text: 'Не доставлено',
                                renderer: function (v) {
                                    if (v && v.length) {
                                        return v.join(', ');
                                    }
                                }
                            },
                            {
                                dataIndex: 'Deleted',
                                text: 'Помечено на удаление',
                                xtype: 'checkcolumn',
                                cls: 'check-button'
                            },
                            {
                                dataIndex: 'DbName',
                                text: 'Наименование БД'
                            }
                            ]
                        }
                    }),
                    oim: ''
                });
            };

        me.FetchMessagesList(node, callback);
    },

    ShowMessage: function (id) {
        if (!id) return;

        var dbKey = ObjX.dbNode.parentNode.get('dbKey');

        if (!dbKey) return;

        AjaxRequest({
            url: 'Mail/GetMailData',
            mask: {
                text: 'Загрузка сообщения',
                id: ObjX.containerMain
            },
            params: {
                dataSource: this.sessionKey,
                id: id
            },
            success: function (res) {
                if (!res) return ExtAlert('#err#Ошибка загрузки письма');
                Ext.create('Meta.Mail.FormReadMessage', {
                    config: {
                        data: res
                    },
                    listeners: {
                        render: function () {
                            this.Toolbar.hide();
                        }
                    }
                });
            }
        });
    },

    FetchMessagesList: function (node, cb) {
        var me = this,
            mask = null;

        if (node) {
            me.MaskNode(node, 'загрузка списка сообщений');
        } else {
            mask = {
                text: 'Обновление списка сообщений',
                id: ObjX.containerMain
            };
        }

        AjaxRequest({
            url: 'DbList/LoadMessageListOutcome',
            mask: mask,
            params: {
                key: node || ObjX.dbNode ? (node || ObjX.dbNode).parentNode.get('dbKey') : null
            },
            success: function (res) {
                node && me.UnmaskNode(node);
                cb && cb.call && cb(res);
            }
        });
    },

    RenderMessageList: function (p) {
        var me = this,
            removeMail = function (params) {
                params.key = objx.dbNode.parentNode.get('dbKey');
                AjaxRequest({
                    url: 'DbList/RemoveMail',
                    mask: {
                        text: 'Удаление сообщений',
                        id: objx.containerMain
                    },
                    params: params,
                    success: list.Refresh
                });
            },
            getSelectionIds = function () {
                var selected = GetSelected('ID');

                if (!selected || !selected.length) return ExtAlert('#msg#Выберите сообщения');

                return selected.map(function (rec) {
                    return {
                        ID: rec.ID
                    };
                });
            },
            tbar = Ext.create('Ext.toolbar.Toolbar', {
                dock: 'top',
                items: [
                    {
                        iconCls: 'ks-icon-open',
                        tooltip: 'Открыть',
                        handler: function () {
                            var selection = getSelectionIds()[0];
                            selection.ID && me.ShowMessage(selection.ID);
                        }
                    },
                    {
                        iconCls: 'ks-icon-delete',
                        tooltip: 'Пометить на удаление',
                        handler: function () {
                            var ids = getSelectionIds();
                            if (ids) {
                                removeMail({
                                    messages: ids,
                                    markDel: true
                                });
                            }
                        }
                    },
                    {
                        iconCls: 'ks-icon-delete_from_server',
                        tooltip: 'Удалить',
                        handler: function () {
                            var ids = getSelectionIds();
                            if (ids) {
                                removeMail({
                                    messages: ids,
                                    markDel: false
                                });
                            }
                        }
                    },
                    {
                        iconCls: 'ks-icon-refresh',
                        tooltip: 'Обновить',
                        handler: function () {
                            list.Refresh();
                        }
                    }
                ]
            }),
            objx = ObjX,
            list = objx.List = Ext.create('Core.List', Ext.apply(p, {
                callback: function () {
                    objx.grid.addDocked(tbar);
                }
            }));

        list.Refresh = function () {
            var cb = function (res) {
                list.localData = res.Messages;
                list.BindStore();
            };
            me.FetchMessagesList(null, cb);
        };
    },

    TurnMailServer: function (mode) {
        var me = this,
            node = me.GetActiveDbNode();

        me.MaskNode(node, (mode ? 'Включение' : 'Отключение') + ' почтового сервера');
        AjaxRequest({
            url: 'DbList/TurnMailServer',
            params: {
                key: me.sessionKey,
                mode: mode
            },
            success: function (res) {
                me.UnmaskNode(node);
                var serverIsOff = res.Fail,
                    text = node.get('text').split('(')[0];
                node.set('text', text + (serverIsOff ? '(отключена)' : '(включена)'));
                node.get('INode').LinkedServer = serverIsOff ? null : '•';
                me.RefreshBranch(node);
            }
        });
    },

    TurnOffMailServer: function () {
        this.TurnMailServer(false);
    },

    TurnSwitchMailServer: function () {
        this.TurnMailServer(true);
    },

    SetTraceStatus: function () {
        var me = this,
            node = me.GetActiveDbNode();

        if (!node) return;

        var menuItem = node.get('extMenu').find(function (item) {
            return item.code === 'Trace';
        });

        if (!menuItem) return;

        var newStatus = !menuItem.checked;
        
        Ext.Msg.show({
            title: 'Подтверждение',
            message: (newStatus ? 'Включить' : 'Выключить') + ' трассировку взаимоблокировок?',
            buttons: Ext.Msg.YESNO,
            icon: Ext.Msg.QUESTION,
            fn: function (btn) {
                if (btn === 'yes') {
                    me.MaskNode(node, 'Перелючение трассировки');
                    AjaxRequest({
                        url: 'DbList/ToggleTraceStatus',
                        params: {
                            key: me.sessionKey,
                            newStatus: newStatus
                        },
                        success: function () {
                            menuItem.checked = newStatus;
                            me.UnmaskNode(node);
                        }
                    });
                }
            }
        });
    },

    ShowServerErrorLog: function (win) {
        var me = this;

        AjaxRequest({
            url: 'DbList/LoadCurrentServerErrorLog',
            mask: {
                text: 'Загрузка журнала',
                id: win
            },
            params: {
                key: me.sessionKey
            },
            success: function (res) {
                if (res && res.Data && res.Data.Tables[0]) {
                    var text = res.Data.Tables[0].Rows.join('\r\n');

                    ExtUtils.ShowWindow({
                        title: 'Журнал ошибок',
                        style: {
                            whiteSpace: 'pre-wrap'
                        },
                        scrollable: 'y',
                        width: 'auto',
                        height: 'auto',
                        buttons: [
                            {
                                text: 'Сохранить',
                                handler: function () {
                                    FileUtils.SaveText(text, 'server_error_log_' + new Date().toLocaleDateString() + '.txt');
                                }
                            },
                            {
                                text: 'Закрыть',
                                handler: function () {
                                    this.up('window').close();
                                }
                            }
                        ],
                        html: Ext.htmlEncode(text)
                    });
                }
            }
        });
    },

    SaveErrorLogToFile: function (win, cb) {
        var me = this;
        
        AjaxRequest({
            url: 'DbList/LoadAllServerErrorLog',
            mask: {
                text: 'Загрузка журнала',
                id: win
            },
            params: {
                key: me.sessionKey
            },
            success: function (res) {
                if (res && res.Data && res.Data.Tables[0]) {
                    var text = res.Data.Tables[0].Rows.join('\r\n'),
                        filename = 'server_error_log_all_' + new Date().toLocaleDateString() + '.txt';
                    

                    FileUtils.SaveText(text, 'server_error_log_all_' + filename);

                    if (cb && cb.call) {
                        cb(filename);
                        win.close();
                    }
                }
            }
        });
    },

    SendServerErrorLog: function (p) {
        var recipient = p.recipient,
            subject = p.subject,
            message = p.message,
            filename = p.filename;

        ExtAlert('#msg#Обязательно прикрепите к письму файл <br> c логами «' + filename + '» из папки загрузок Вашего браузера');
        
        ExtUtils.SilentAnchorClick('mailto:' + recipient + '?subject= ' + subject + '&body=' + message);
    },

    ServerErrorLog: function () {
        var me = this;

        ExtUtils.ShowWindow({
            title: 'Журнал ошибок',
            modal: true,
            width: 550,
            height: 450,
            viewModel: {
                data: {
                    recipient: 'budjet@keysystems.ru',
                    subject: 'Журнал ошибок',
                    message: '',
                    attachErrorLog: true
                }
            },
            items: {
                xtype: 'form',
                title: 'Отправить отчет об ошибках',
                defaultType: 'textfield',
                bodyPadding: 8,
                border: 0,
                margin: 0,
                layout: {
                    type: 'vbox',
                    align: 'stretch'
                },
                defaults: {
                    labelWidth: 100,
                    labelAlign: 'left'
                },
                items: [
                    {
                        fieldLabel: 'Кому',
                        allowBlank: false,
                        bind: '{recipient}'
                    },
                    {
                        fieldLabel: 'Тема',
                        allowBlank: false,
                        bind: '{subject}'
                    },
                    {
                        xtype: 'textarea',
                        fieldLabel: 'Сообщение',
                        allowBlank: false,
                        labelAlign: 'top',
                        flex: 1,
                        emptyText: 'Опишите ситуацию, при которой возникает ошибка, чтобы наши специалисты смогли её воспроизвести',
                        bind: '{message}'
                    },
                    {
                        xtype: 'fieldcontainer',
                        layout: {
                            type: 'hbox',
                            align: 'center'
                        },
                        items: [
                            {
                                xtype: 'checkbox',
                                boxLabel: 'Прикрепить журнал сервера БД',
                                allowBlank: false,
                                bind: '{attachErrorLog}'
                            },
                            {
                                xtype: 'container',
                                border: 0,
                                flex: 1
                            },
                            {
                                xtype: 'button',
                                text: 'Просмотр',
                                handler: function () {
                                    me.ShowServerErrorLog(this.up('window'));
                                }
                            }
                        ]
                    }
                ]
            },
            buttons: [
                {
                    xtype: 'button',
                    text: 'Отправить',
                    bind: {
                        disabled: '{!attachErrorLog}'
                    },
                    handler: function () {
                        var win = this.up('window'),
                            form = win.down('form');
                        if (form.isValid()) {
                            me.SaveErrorLogToFile(win, function (filename) {
                                var data = win.getViewModel().getData();
    
                                data.filename = filename;
    
                                me.SendServerErrorLog(data);
                            });
                        } else {
                            ExtAlert('#msg#Заполните все поля');
                        }
                    }
                },
                {
                    xtype: 'button',
                    text: 'Сохранить',
                    bind: {
                        disabled: '{!attachErrorLog}'
                    },
                    handler: function () {
                        me.SaveErrorLogToFile(this.up('window'));
                    }
                },
                {
                    xtype: 'button',
                    text: 'Закрыть',
                    handler: function () {
                        this.up('window').close();
                    }
                }
            ]
        });
    }
});