﻿class CadesCryptoModule {
    constructor() {
        // Инициализация модуля взаимодействия с плагином
        // cadesplugin_api.js с пакета http://npm.keysystems.ru/crypto-pro-cadesplugin/-/crypto-pro-cadesplugin-1.1.1.tgz
        new CadesPluginApi();
    }

    CheckConnection() {
        const me = this;

        return window.cadesplugin.async_spawn(function* () {
            yield window.cadesplugin.catch(function (e) {
                    Ext.create(me.getInstallDialog());

                    // Сбрасываем настройки плагина, для повторной инициализации
                    // Но всё равно не помогает, пока не обновить страницу, плагин не увидит
                    KS.XCrypt.cadesModule = null;

                    throw new Error('#install#' + e);
                });

            const providerName = 'Crypto-Pro GOST R 34.10-2012 Cryptographic Service Provider';
            const providerType = '80';

            try {
                const oAbout =
                    yield window.cadesplugin.CreateObjectAsync(
                        'CAdESCOM.About'
                    );
                const oVersion = yield oAbout.CSPVersion(
                    providerName,
                    parseInt(providerType, 10)
                );

                /* const Minor = */
                yield oVersion.MinorVersion;
                /* const Major = */
                yield oVersion.MajorVersion;
                /* const Build = */
                yield oVersion.BuildVersion;
                /* const Version = */
                yield oVersion.toString();
            } catch (e) {
                Ext.create(me.getInstallDialog());

                const err = window.cadesplugin.getLastError(e);

                if (err.indexOf('0x80090019') + 1) {
                    throw new Error("#msg#Указанный CSP не установлен");
                }

                throw e;
            }
        });
    }

    getInstallDialog() {
        var html = this.getResponseHtml();
        return Ext.create("Ext.window.Window", {
            width: 560,
            bodyPadding: 25,
            title: "Список сертификатов",
            modal: true,
            layout: {
                type: "vbox",
                align: "stretch"
            },
            items: [
                {
                    xtype: "panel",
                    border: 0,
                    scrollable: "y",
                    html: html,
                    margin: "0 0 0 0",
                    flex: 5
                },
                {
                    xtype: "panel",
                    layout: "hbox",
                    margin: "0 -4",
                    border: 0,
                    defaults: {
                        iconCls: 'ks-icon-download_icon',
                        textAlign: 'center',
                        margin: '0 4'
                    },
                    items:  {
                        xtype: "button",
                        text: "Перейти по ссылке",
                        href: 'https://www.cryptopro.ru/products/cades/plugin',
                        scale: 'large',
                    }
                }
            ],
            buttons: [{
                text: "Отмена",
                handler: function () {
                    this.up("window").close();
                }
            }]
        }).show();
    }

    getResponseHtml() {
        if (!this.responseHtml) {
            this.responseHtml = this.createResponseHtml();
        }
        return this.responseHtml;
    }

    createResponseHtml() {
        var style = "<style>" +
            ".kss-install-window h2 { color: #000; }" +
            ".kss-install-window__title {margin-top: 0;font-size: 18px;font-weight: 400;line-height: 1.33;}" +
            ".kss-install-window__logo-text {font-size:18px;margin:0;flex-grow:1;}" +
            ".kss-install-window__text {margin-bottom: 16px;}" +
            "</style>";

        var html = "<div class=\"kss-install-window\">" +
            "<h2 class=\"kss-install-window__title\">Для получения списка сертификатов необходимо установить/запустить приложение для подписи документов:</h2>" +
            "<h2 class=\"kss-install-window__logo-text\">КриптоПро ЭЦП Browser plug-in</h2>" +
            "</div>" +
            "<div class=\"kss-install-window__text\">" +
            "<p><sup>*</sup>После установки/включения модуля необходимо обновить страницу</p>" +
            "</div>";

        return style + html;
    }

    /**
     *
     * @param sHashValue Хэш-значение, которым следует проинициализировать объект. Должно быть представлено в виде строки шестнадцатеричных цифр.
     * @param certSerial
     * @param pubkeyalgoid
     * @returns {Promise<unknown>}
     * @constructor
     */
    async SignHash(sHashValue, certSerial, pubkeyalgoid) {
        const me = this;

        return new Promise(function(resolve, reject) {
            cadesplugin.async_spawn(function* () {
                yield me.CheckConnection()

                let hashAlg;

                if (pubkeyalgoid == "1.2.643.7.1.1.1.1") {
                    hashAlg = cadesplugin.CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256;
                } else if (pubkeyalgoid == "1.2.643.7.1.1.1.2") {
                    hashAlg = cadesplugin.CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_512;
                }

                let oCertificate = yield me.GetCertificate(certSerial);

                const oSigner = yield cadesplugin.CreateObjectAsync("CAdESCOM.CPSigner");
                yield oSigner.propset_Certificate(oCertificate);
                yield oSigner.propset_CheckCertificate(true);

                const oHashedData = yield cadesplugin.CreateObjectAsync("CAdESCOM.HashedData");
                yield oHashedData.propset_Algorithm(hashAlg);
                yield oHashedData.propset_DataEncoding(cadesplugin.CADESCOM_BASE64_TO_BINARY)
                yield oHashedData.SetHashValue(sHashValue);

                const oSignedData = yield cadesplugin.CreateObjectAsync("CAdESCOM.CadesSignedData");
                //var sSignedMessage = yield oSignedData.SignHash(oHashedData, oSigner, CADESCOM_CADES_BES);
                //return args[0](sSignedMessage);
                let sSignedMessage;

                try {
                    sSignedMessage = yield oSignedData.SignHash(oHashedData, oSigner, cadesplugin.CADESCOM_CADES_BES);
                } catch (e) {
                    const err = cadesplugin.getLastError(e);

                    if (err.indexOf("0x8010006E") + 1) {
                        throw new CancelError();
                    }
                    else if (err.indexOf("0x80090014") + 1) {
                        throw new Error('#err#Отсутствует ссылка на закрытый ключ (0x80090014)');
                    }
                    else if (err.indexOf('0x80090016') + 1) {
                        throw new Error('#err#Ошибка обращения к контейнеру закрытого ключа. Набор ключей не существует (0x80090016)');
                    }
                    else if (err.indexOf("0x8010006E") + 1) {
                        throw new Error('#err#Операция была отменена пользователем (0x8010006E)');
                    }
                    else {
                        throw new Error("Не удалось создать подпись. Ошибка: " + err);
                    }
                }

                // Создаем объект CAdESCOM.CadesSignedData
                // const oSignedData2 = yield cadesplugin.CreateObjectAsync("CAdESCOM.CadesSignedData");

                // Проверяем подпись
                // try {
                //     yield oSignedData2.VerifyHash(oHashedData, sSignedMessage, cadesplugin.CADESCOM_CADES_BES);
                // } catch (e) {
                //     const err = cadesplugin.getLastError(e);
                //
                //     throw new Error('Не удалось проверить подпись. Ошибка: ' + err);
                // }

                return sSignedMessage;
            })
                .then(resolve)
                .catch(reject)
        });
    }

    async SignText(dataForSign, certSerial, convert) {
        const me = this;

        return new Promise(function(resolve, reject) {
            cadesplugin.async_spawn(function* () {
                yield me.CheckConnection()

                let oCertificate = yield me.GetCertificate(certSerial);

                const oSigner = yield cadesplugin.CreateObjectAsync("CAdESCOM.CPSigner");
                yield oSigner.propset_Certificate(oCertificate);
                yield oSigner.propset_CheckCertificate(true);
                //yield oSigner.propset_TSAAddress("http://cryptopro.ru/tsp/");

                const oSignedData = yield cadesplugin.CreateObjectAsync("CAdESCOM.CadesSignedData");
                yield oSignedData.propset_ContentEncoding(cadesplugin.CADESCOM_BASE64_TO_BINARY);
                yield oSignedData.propset_Content(dataForSign);

                try {
                    return yield oSignedData.SignCades(oSigner, cadesplugin.CADESCOM_CADES_BES, true);
                } catch (e) {
                    const err = cadesplugin.getLastError(e);

                    if (err.indexOf("0x8010006E") + 1) {
                        throw new CancelError();
                    }
                    else if (err.indexOf("0x80090014") + 1) {
                        throw new Error('#err#Отсутствует ссылка на закрытый ключ (0x80090014)');
                    }
                    else if (err.indexOf('0x80090016') + 1) {
                        throw new Error('#err#Ошибка обращения к контейнеру закрытого ключа. Набор ключей не существует (0x80090016)');
                    }
                    else if (err.indexOf("0x8010006E") + 1) {
                        throw new Error('#err#Операция была отменена пользователем (0x8010006E)');
                    }
                    else {
                        throw new Error("Не удалось создать подпись. Ошибка: " + err);
                    }
                }
            })
                .then(resolve)
                .catch(reject)
        });
    }    

    Verify(sSignedMessage) {
        return cadesplugin.async_spawn(function* (args) {
            const oSignedData = yield cadesplugin.CreateObjectAsync("CAdESCOM.CadesSignedData");
            
            try {
                return yield oSignedData.VerifyCades(sSignedMessage, cadesplugin.CADESCOM_CADES_BES);
            }
            catch (e) {
                const err = cadesplugin.getLastError(e);
                
                throw new Error('Не удалось проверить подпись. Ошибка: ' + err);
            }            
        });        
    }

    /**
     * Получим все сертификаты из личного хранилища
     * @param {CadesCertificateParams} Params
     * @returns {Promise<unknown>}
     */
    async GetCertificates(Params = new CadesCertificateParams({})) {
        const me = this;

        return new Promise(function(resolve, reject) {
            cadesplugin.async_spawn(function* () {
                yield me.CheckConnection();

                let oStore;

                try {
                    oStore = yield cadesplugin.CreateObjectAsync("CAdESCOM.Store");
                }
                catch (e) {
                    const err = cadesplugin.getLastError(e);

                    if (err.indexOf("0x000004C7") + 1) {
                        throw new CancelError();
                    }

                    throw e;
                }

                yield oStore.Open(cadesplugin.CAPICOM_CURRENT_USER_STORE, cadesplugin.CAPICOM_MY_STORE,
                    cadesplugin.CAPICOM_STORE_OPEN_MAXIMUM_ALLOWED);
                let certsList = [];
                let certs = yield oStore.Certificates;
                let certCnt = yield certs.Count;

                for (let i = 1; i <= certCnt; i++) {
                    let cert, certIsValid;

                    try {
                        cert = yield certs.Item(i);
                    } catch (e) {
                        const err = cadesplugin.getLastError(e);

                        throw new Error("Не удалось получить сертификат. Ошибка: " + err);
                    }
                    //если попадется сертификат с неизвестным алгоритмом
                    //тут будет исключение. В таком сертификате просто пропускаем такое поле
                    try {
                        const Validator = yield cert.IsValid();
                        certIsValid = yield Validator.Result;
                    } catch(e) {
                        console.error(e);

                        certIsValid = false;
                    }

                    if (certIsValid) {
                        const parsed_certificate = yield me.parseCertificateData(cert, Params);
                        certsList.push(parsed_certificate);
                    }
                }
                yield oStore.Close();

                return certsList;
            })
                .then(resolve)
                .catch(reject)
        });
    }

    /**
     *
     * @param certificate {CAPICOM.Certificate}
     * @param {CadesCertificateParams} Params
     */
    parseCertificateData(certificate, 
                         Params = new CadesCertificateParams({})
    ){
        const me = this;

        return new Promise(function (resolve, reject) {
            window.cadesplugin
                .async_spawn(function* () {
                    let filterreason, warnreason;

                    const hasprivatekey = yield certificate.HasPrivateKey();
                    const serial = yield certificate.SerialNumber;
                    const notAfterStr = yield certificate.ValidToDate;
                    const notAfter = new Date(notAfterStr);
                    const notBeforeStr = yield certificate.ValidFromDate;
                    const notBefore = new Date(notBeforeStr);
                    const subjectStr = yield certificate.SubjectName;
                    const subjSn = me.GetMarkerValue(subjectStr, 'SN');
                    const subjG = me.GetMarkerValue(subjectStr, 'G');
                    const subjCn = me.GetMarkerValue(subjectStr, 'CN');
                    let fio;
                    if (subjSn && subjG) {
                        fio = subjSn + ' ' + subjG;
                    } else {
                        fio = subjCn;
                    }
                    const issuerStr = yield certificate.IssuerName;

                    // Оригинальные проверки в GoLang в ядре ЭП - CryptoModule
                    // FilterByPrivateKey
                    if (!hasprivatekey) {
                        filterreason = "Отсутствует ссылка на закрытый ключ.";
                    } else if (Params != null) {
                        filterreason = me.FilterByTime(
                            Params.SecretKeyValidity,
                            notBefore,
                            notAfter
                        );
                        if (!filterreason) {
                            filterreason = me.FilterBySerial(serial, Params.KeyId);
                        }
                        if (!filterreason) {
                            filterreason = me.FilterBySubject(
                                subjectStr,
                                Params.KeyId,
                                subjSn,
                                subjG
                            );
                        }
                        if (!filterreason) {
                            filterreason = me.FilterByIssuer(
                                issuerStr,
                                Params.IssuerFilter
                            );
                        }
                        if (
                            Params.FilterOid !== null &&
                            Params.FilterOid > ''
                        ) {
                            // FilterByOID
                            warnreason = `$Проверка по OID для [Cades Plugin] не реализована!`;
                        }
                        if (
                            !filterreason &&
                            Params.NumDaysWarnBeforeCertEnd !== null &&
                            Params.NumDaysWarnBeforeCertEnd > 0
                        ) {
                            // WarnByDays
                            const ret = me.WarnByDays(
                                Params.NumDaysWarnBeforeCertEnd,
                                notBefore,
                                notAfter
                            );
                            if (ret.warnreason) {
                                warnreason =
                                    (warnreason ? warnreason + '\r\n' : '') +
                                    ret.warnreason;
                            }
                            if (ret.filterreason) {
                                filterreason = ret.filterreason;
                            }
                        }
                    }

                    const b64 = yield certificate.Export(
                        window.cadesplugin.CADESCOM_ENCODE_BASE64
                    );

                    const pubKey = yield certificate.PublicKey();
                    const algo = yield pubKey.Algorithm;
                    const pubKeyOid = yield algo.Value;
                    const pubKeyName = yield algo.FriendlyName;

                    const {
                        signaturealgoid,
                        hashalgoid,
                        signaturealgname,
                        hashalgname
                    } = me.GetSigHashOidsNamesForExchKeyOid(pubKeyOid);

                    return {
                        subjectid: null,
                        serial,
                        subject: subjectStr,
                        subjectx500: {
                            CN: subjCn,
                            SN: subjSn,
                            G: subjG,
                            T: me.GetMarkerValue(subjectStr, 'T'),
                            E: me.GetMarkerValue(subjectStr, 'E'),
                            O: me.GetMarkerValue(subjectStr, 'O'),
                            OU: me.GetMarkerValue(subjectStr, 'OU'),
                            S: me.GetMarkerValue(subjectStr, 'S'),
                            L: me.GetMarkerValue(subjectStr, 'L'),
                            STREET: me.GetMarkerValue(subjectStr, 'STREET'),
                            C: me.GetMarkerValue(subjectStr, 'C'),
                            INN:
                                me.GetMarkerValue(subjectStr, 'INN') ||
                                me.GetMarkerValue(subjectStr, 'ИНН') ||
                                me.GetMarkerValue(
                                    subjectStr,
                                    '1.2.643.3.131.1.1'
                                ),
                            SNILS:
                                me.GetMarkerValue(subjectStr, 'SNILS') ||
                                me.GetMarkerValue(subjectStr, 'СНИЛС') ||
                                me.GetMarkerValue(subjectStr, '1.2.643.100.3'),
                            OGRN:
                                me.GetMarkerValue(subjectStr, 'OGRN') ||
                                me.GetMarkerValue(subjectStr, 'ОГРН') ||
                                me.GetMarkerValue(subjectStr, '1.2.643.100.1'),
                            OGRNIP:
                                me.GetMarkerValue(subjectStr, 'OGRNIP') ||
                                me.GetMarkerValue(subjectStr, 'ОГРНИП') ||
                                me.GetMarkerValue(subjectStr, '1.2.643.100.5'),
                            FIO: fio
                        },
                        issuer: issuerStr,
                        issuerx500: {
                            CN: me.GetMarkerValue(issuerStr, 'CN'),
                            SN: me.GetMarkerValue(issuerStr, 'SN'),
                            G: me.GetMarkerValue(issuerStr, 'G'),
                            T: me.GetMarkerValue(issuerStr, 'T'),
                            E: me.GetMarkerValue(issuerStr, 'E'),
                            O: me.GetMarkerValue(issuerStr, 'O'),
                            OU: me.GetMarkerValue(issuerStr, 'OU'),
                            S: me.GetMarkerValue(issuerStr, 'S'),
                            L: me.GetMarkerValue(issuerStr, 'L'),
                            STREET: me.GetMarkerValue(issuerStr, 'STREET'),
                            C: me.GetMarkerValue(issuerStr, 'C'),
                            INN:
                                me.GetMarkerValue(issuerStr, 'INN') ||
                                me.GetMarkerValue(issuerStr, 'ИНН') ||
                                me.GetMarkerValue(
                                    issuerStr,
                                    '1.2.643.3.131.1.1'
                                ),
                            SNILS:
                                me.GetMarkerValue(issuerStr, 'SNILS') ||
                                me.GetMarkerValue(issuerStr, 'СНИЛС') ||
                                me.GetMarkerValue(issuerStr, '1.2.643.100.3'),
                            OGRN:
                                me.GetMarkerValue(issuerStr, 'OGRN') ||
                                me.GetMarkerValue(issuerStr, 'ОГРН') ||
                                me.GetMarkerValue(issuerStr, '1.2.643.100.1'),
                            OGRNIP:
                                me.GetMarkerValue(issuerStr, 'OGRNIP') ||
                                me.GetMarkerValue(issuerStr, 'ОГРНИП') ||
                                me.GetMarkerValue(issuerStr, '1.2.643.100.5')
                        },
                        notbefore: notBeforeStr,
                        notafter: notAfterStr,
                        pubkeyalgoid: pubKeyOid,
                        pubkeyalgname: pubKeyName,
                        signaturealgoid,
                        signaturealgname,
                        hasprivatekey,
                        bytesb64: b64.replace(/\s/g, ''),
                        filterreason,
                        message: warnreason,
                        // provname: providerName,
                        // provtype: providerType,
                        // containername: containerName,
                        hashalgoid,
                        hashalgname,
                        cert: certificate
                    };
                })
                .then(resolve)
                .catch(reject);
        });
    }

    /**
     * CAPICOM.Certificate
     * @param {String} serialNumber - Серийный номер с решеткой
     */
    async GetCertificate(serialNumber) {
        const me = this;

        serialNumber = (
            serialNumber.startsWith('#')
                ? serialNumber.substring(1)
                : serialNumber
        ).toLowerCase();

        return new Promise(function (resolve, reject) {
            window.cadesplugin
                .async_spawn(function* () {
                    yield me.CheckConnection();

                    let oStore;

                    try {
                        oStore =
                            yield window.cadesplugin.CreateObjectAsync(
                                'CAdESCOM.Store'
                            );
                    } catch (e) {
                        const err = window.cadesplugin.getLastError(e);

                        if (err.indexOf('0x000004C7') + 1) {
                            throw new window.KS.CryptoModule.Errors.Cancel();
                        }

                        throw e;
                    }

                    yield oStore.Open(
                        window.cadesplugin.CAPICOM_CURRENT_USER_STORE,
                        window.cadesplugin.CAPICOM_MY_STORE,
                        window.cadesplugin.CAPICOM_STORE_OPEN_MAXIMUM_ALLOWED
                    );

                    const certs = yield oStore.Certificates;
                    const certCnt = yield certs.Count;
                    let foundCert;

                    for (let i = 1; i <= certCnt; i++) {
                        let cert;

                        try {
                            cert = yield certs.Item(i);
                        } catch (e) {
                            const err = window.cadesplugin.getLastError(e);

                            throw new Error(
                                `Не удалось получить сертификат. Ошибка: ${err}`
                            );
                        }

                        const checkSerialNumber =
                            (yield cert.SerialNumber).toLowerCase();

                        if (serialNumber === checkSerialNumber) {
                            foundCert = cert;
                            break;
                        }
                    }
                    yield oStore.Close();

                    return foundCert;
                })
                .then(resolve)
                .catch(reject);
        });
    }

    FilterByIssuer(issuerStr, issuerFilter) {
        const lowerIssuer = issuerStr.toLowerCase();
        if (issuerFilter !== null && issuerFilter > '') {
            const checkIssuersStr = issuerFilter
                .toLowerCase()
                .replaceAll(',', ';');
            const checkIssuers = checkIssuersStr.split(';');
            for (const checkIssuer of checkIssuers) {
                const checkSubjectTrimmed = checkIssuer.trim();
                if (lowerIssuer.indexOf(checkSubjectTrimmed) > -1) {
                    return '';
                }
            }
            return `Издатель [${lowerIssuer}] не удовлетворяет ни одному из включений: [${checkIssuers.join(', ')}]`;
        }
        return '';
    }

    FilterBySubject(subjectStr, KeyId, subjSn, subjG) {
        let lowerSubject = subjectStr.toLowerCase();
        if (KeyId !== null && KeyId > '' && !KeyId.startsWith('#')) {
            const checkSubjectsStr = KeyId.toLowerCase().replaceAll(',', ';');
            const checkSubjects = checkSubjectsStr.split(';');
            if (subjSn && subjG) {
                lowerSubject += ' ' + subjSn.trim() + ' ' + subjG.trim() + ' ';
            }
            for (const checkSubject of checkSubjects) {
                const checkSubjectTrimmed = checkSubject.trim();
                if (lowerSubject.indexOf(checkSubjectTrimmed) > -1) {
                    return '';
                }
            }
            return `Субъект [${lowerSubject}] не удовлетворяет включениям: [${checkSubjects.join(', ')}]`;
        }
        return '';
    }

    FilterBySerial(serial, KeyId) {
        const lowerSerial = serial.toLowerCase();
        if (KeyId !== null && KeyId > '' && KeyId.startsWith('#')) {
            const checkSerialsStr = KeyId.toLowerCase()
                .replaceAll(',', ';')
                .replaceAll('#', '')
                .replaceAll(' ', '');
            const checkSerials = checkSerialsStr.split(';');
            for (const checkSerial of checkSerials) {
                const checkSerialTrimmed = checkSerial.trim();
                if (lowerSerial === checkSerialTrimmed) {
                    return '';
                }
            }
            return `Серийный номер [${serial}] не удовлетворяет ни одному из значений: [${checkSerials.join(',')}]`;
        }
        return '';
    }

    /**
     * @param {number} dayswarn
     * @param {Date} notBefore
     * @param {Date} notAfter
     */
    WarnByDays(dayswarn, notBefore, notAfter) {
        let warnreason, filterreason;
        if (dayswarn > 0) {
            const prefix = 'Срок действия сертификата';
            const d = Math.round(
                dayswarn -
                (new Date() -
                    Ext.Date.add(notAfter, Ext.Date.DAY, -dayswarn)) /
                (1000 * 3600 * 24)
            );
            if (d <= dayswarn) {
                let errMsg = '';
                let msg = '';
                let daysStr = '';
                if (d < 0) {
                    errMsg = `${prefix} истёк.`;
                } else if (d === 0) {
                    msg = `заканчивается cегодня.`;
                } else if (d === 1) {
                    msg = `заканчивается завтра.`;
                } else {
                    if (d <= 4) {
                        daysStr = 'дня';
                    } else {
                        daysStr = 'дней';
                    }
                    msg = `заканчивается.\r\nДо окончания срока действия осталось ${d} ${daysStr}.`;
                }

                filterreason = errMsg;
                if (!filterreason) {
                    warnreason = `${prefix} ${msg}`;
                }
            }
        }
        return {
            filterreason,
            warnreason
        };
    }

    FilterByTime(secretKeyValidity, notBefore, notAfter) {
        // Extenstions не доступно, нет возможности вытащить даты закрытого ключа, сделаем только по дате открытого
        // const extensions = yield certificate.Extensions;
        let filterreason = '';
        let settingUsed = '';
        if (secretKeyValidity > 0) {
            const teCand = Ext.Date.add(
                notBefore,
                Ext.Date.DAY,
                secretKeyValidity
            );
            if (teCand < notAfter) {
                notAfter = teCand;
            }
            settingUsed = ` По настройке: ${secretKeyValidity} дня (-ей) от даты начала по [${notAfter.toLocaleString()}]`;

            const now = new Date();
            if (!(notBefore < now && now < notAfter)) {
                filterreason = `Срок действия сертификата истёк.${settingUsed}`;
            }
        }
        return filterreason;
    }

    // Взято из Keysystems.Security.Signing\CertificateHelperBase.cs
    GetMarkerValue(str = '', substr) {
        const markerRegExp = new RegExp(
            '(?<=(?: |^)' + substr + '\\=).*?(?=,\\s*[\\w\\. А-Яа-яЁё]+\\=|$)'
        );

        const matchMarkerValue = str.match(markerRegExp);
        let markerValue = '';

        if (matchMarkerValue && matchMarkerValue.length === 1) {
            markerValue = this.TrimQuotesIfNeed(matchMarkerValue[0]);
        }

        return markerValue;
    }

    // Взято из Keysystems.CryptoModule (GoLang)
    TrimQuotesIfNeed(str) {
        if (str[0] == '"' && str[str.length - 1] == '"') {
            return str.substring(1, str.length - 2).replace(/""/g, '"');
        }

        return str;
    }

    // Взято из Keysystems.CryptoModule (GoLang)
    GetSigHashOidsNamesForExchKeyOid(pubKeyOid) {
        const me = this;

        switch (pubKeyOid) {
            case this.oidRSA:
                return {
                    signaturealgoid: me.oidRSA,
                    hashalgoid: me.oidSHA1,
                    signaturealgname: 'RSA',
                    hashalgname: 'SHA-1'
                };
            case this.oidCP_GOST_R3410EL:
                return {
                    signaturealgoid: me.oidCP_GOST_R3411_R3410EL,
                    hashalgoid: me.oidCP_GOST_R3411,
                    signaturealgname: 'ГОСТ Р 34.11-94/ГОСТ Р 34.10-2001',
                    hashalgname: 'ГОСТ Р 34.11-94'
                };
            case this.oidCP_GOST_R3410_12_256:
                return {
                    signaturealgoid: me.oidCP_GOST_R3411_12_256_R3410,
                    hashalgoid: me.oidCP_GOST_R3411_12_256,
                    signaturealgname: 'ГОСТ Р 34.11-2012/ГОСТ Р 34.10-2012 256',
                    hashalgname: 'ГОСТ Р 34.10-2012 256'
                };
            case this.oidCP_GOST_R3410_12_512:
                return {
                    signaturealgoid: me.oidCP_GOST_R3411_12_512_R3410,
                    hashalgoid: me.oidCP_GOST_R3411_12_512,
                    signaturealgname: 'ГОСТ Р 34.11-2012/ГОСТ Р 34.10-2012 512',
                    hashalgname: 'ГОСТ Р 34.11-2012 512'
                };
        }
    }

    // Функция хэширования 1.3.14.3.2.26 - SHA-1 hash algorithm
    oidSHA1 = '1.3.14.3.2.26';
    // Алгоритм цифровой подписи 1.2.840.113549.1.1.1 - RSA encryption
    oidRSA = '1.2.840.113549.1.1.1';

    // Функция хэширования ГОСТ Р 34.11-94
    oidCP_GOST_R3411 = '1.2.643.2.2.9';
    // Алгоритм цифровой подписи ГОСТ Р 34.10-2001
    oidCP_GOST_R3411_R3410EL = '1.2.643.2.2.3';
    // Алгоритм ГОСТ Р 34.10-2001.используемый при экспорте/импорте ключей
    oidCP_GOST_R3410EL = '1.2.643.2.2.19';

    // Функция хэширования ГОСТ Р 34.11-2012.длина выхода 256 бит
    oidCP_GOST_R3411_12_256 = '1.2.643.7.1.1.2.2';
    // Алгоритм цифровой подписи ГОСТ Р 34.10-2012 для ключей длины 256 бит
    oidCP_GOST_R3411_12_256_R3410 = '1.2.643.7.1.1.3.2';
    // Алгоритм ГОСТ Р 34.10-2012 для ключей длины 256 бит.используемый при экспорте/импорте ключей
    oidCP_GOST_R3410_12_256 = '1.2.643.7.1.1.1.1';

    // Функция хэширования ГОСТ Р 34.11-2012.длина выхода 512 бит
    oidCP_GOST_R3411_12_512 = '1.2.643.7.1.1.2.3';
    // Алгоритм цифровой подписи ГОСТ Р 34.10-2012 для ключей длины 512 бит
    oidCP_GOST_R3411_12_512_R3410 = '1.2.643.7.1.1.3.3';
    // Алгоритм ГОСТ Р 34.10-2012 для ключей длины 512 бит.используемый при экспорте/импорте ключей
    oidCP_GOST_R3410_12_512 = '1.2.643.7.1.1.1.2';
}

class CadesCertificateParams {
    constructor({ FilterOid = null, IssuerFilter = null, SecretKeyValidity = null, NumDaysWarnBeforeCertEnd = null, MchdDatas = null, InnInfos = null }) {
        this.KeyId = FilterOid;
        this.FilterOid = FilterOid;
        this.IssuerFilter = IssuerFilter;
        this.SecretKeyValidity = SecretKeyValidity;
        this.NumDaysWarnBeforeCertEnd = NumDaysWarnBeforeCertEnd;
        this.MchdDatas = MchdDatas;
        this.InnInfos = InnInfos;
    }
}