Ext.define('Svod.UndoManager', {
    alias: 'undo.manager',

    constructor: function (cfg) {
        this.records = [];
        this.undoPosition = -1;
        this.closeCode = cfg.closeCode;
        this.isCaseSensitive = cfg.isCaseSensitive;

        cfg.grid.on('edit', this.onGridEdit, this);
        cfg.grid.getStore().on('remove', this.onGridStoreRemove, this);
    },

    onGridEdit: function (editor, context) {
        var me = this;

        me.addUndoRecord({
            link: KS.Grid.getAnyCase(context.record, me.closeCode),
            colKey: context.field,
            value: context.record.get(context.field),
            originalValue: context.originalValue
        });
    },

    onGridStoreRemove: function (grid, recsToRemove) {
        var me = this,
            deletedSavedRowLinks = [],
            undoPositionShift = 0;

        Ext.each(recsToRemove, function (rec) {
            var link = KS.Grid.getAnyCase(rec, me.closeCode);
            deletedSavedRowLinks.push(link);
        });

        // Забываем изменения, которые были в удаленных строках
        me.records = Ext.Array.filter(me.records, function (rec, index) {
            if (deletedSavedRowLinks.includes(rec.link)) {
                if (index <= me.undoPosition)
                    undoPositionShift++;
                return false;
            }
            return true;
        });

        // Сдвигаем undoPosition
        me.undoPosition = me.undoPosition - undoPositionShift;
    },

    addUndoRecord: function (undoRecord) {
        var records = this.records,
            position = this.undoPosition;

        if (position < records.length - 1) {
            // Забываем изменения, которые были отменены
            records.splice(position + 1, records.length - position + 1);
        }

        records.push(undoRecord);

        this.undoPosition++;
    },

    undo: function (gridStore) {
        var me = this;

        if (me.undoPosition < 0)
            return;

        var currentRec = me.records[me.undoPosition];
        var rec = me.findRecordInStore(gridStore, currentRec.link);

        if (!rec)
            return;

        rec.set(currentRec.colKey, currentRec.originalValue);
        me.undoPosition--;
        return currentRec;
    },

    redo: function (gridStore) {
        var me = this;

        if (me.undoPosition === me.records.length - 1)
            return;

        me.undoPosition++;
        var currentRec = me.records[me.undoPosition];
        var rec = me.findRecordInStore(gridStore, currentRec.link);

        if (!rec) {
            me.undoPosition--;
            return;
        }

        rec.set(currentRec.colKey, currentRec.value);
        return currentRec;
    },

    findRecordInStore: function (store, link) {
        var me = this;

        return me.isCaseSensitive
            ? store.findRecord(me.closeCode, link)
            : store.findRecord(me.closeCode.toLowerCase(), link) ||
              store.findRecord(me.closeCode.toUpperCase(), link);
    }
});