Ext.define('Revizor.RangView', {
	extend: 'Ext.window.Window',
	mixins: ['Keysystems.Base.Abstract'],
	modal: true,
	iconCls: 'x_btn_rang',
	layout: 'border',
	width: 800,
	height: 400,
	minHeight: 200,
	minWidth: 400,
	maximizable: true,
	labelWidth: 200,
	resizable: true,
	formName: 'RangForm2RISK_INFO',
	checkChanges: true,
	//показатели ранжирования
	rangCalcs: [],
	//линк варианта
	variant: 0,
	//флаг - выполняется установка значений по коллекции показателей 
	setValuesInProgress: false,

	constructor: function (cfg) {
		Ext.apply(this, cfg);
		this.callParent(arguments);
		this.rangCalcs = [];
		this.oldData = this.dataCollector();
	},

	/** Сохранить и закрыть  */
	submit: function () {
		let me = this;
		me.saveVariant(() => {
			me.checkChanges = false;
			if (me.ok) me.ok(me.variant);
			me.close();
		});
	},

	/** Применить */
	apply: function () {
		this.saveVariant();
	},

	/** Сохранить вариант */
	saveVariant: function (callback) {
		let me = this;
		if (!me.rangCalcs.length) {
			if (callback) callback();
			return;
		}
		let newData = me.dataCollector();
		if (me.oldData === newData) {
			if (callback) callback();
			return;
		}

		console.log('Сохранение варианта ' + me.variant);
		console.log(me.rangCalcs);

		let link = me.variant ?? 0,
			params = {
				calcs: me.rangCalcs,
				variant: link,
				name: link === 0 ? me.gksd('tempVariantName') : '',
				formName: link === 0 ? me.formName : ''
			};

		const loadMask = new Ext.LoadMask({
			msg: KS.L10n.RiskInfoListView_RangCalcSaving,
			target: me,
			autoShow: true,
			rid: ajaxRequest({
				url: '/SRiskInfo/SaveVariant_A',
				params: {gzipData: SignalR.pack(params)},
				success: function (result) {
					loadMask.destroy();
					if (!result || !result.saved) {
						console.warn('Не удалось сохранить вариант ' + link);
						return false;
					}
					me.variant = result.savedLink;
					me.oldData = me.dataCollector();
					console.log('Сохранение варианта выполнено успешно, savedLink = ' + me.variant);
					if (callback) callback();
				},
				failure: function (response) {
					loadMask.destroy();
					console.error("Error", response.responseText);
				}
			})
		});
	},

	createButtons: function () {
		let me = this;
		return [
			me.dictVariant = me.createDictEdit({
				key: 'RANG_S_VARARAM',
				controlName: 'DICTEDITOR_RangView_dictSVarparam',
				flex: 1,
				fieldLabel: 'Вариант',
				labelWidth: 100,
				code: dnl.S_VARPARAM,
				mode: 'SINGLE',
				parentView: me,
				whereArgs: {
					'FormName': {
						value: me.formName,
						type: 'string'
					}
				},
				inputDicts: [
					Ext.create('InputEditDict', {
						'name': 'FORMNAME',
						'data': me.formName
					})
				],
				ksSetValue: function (v) {
					this.setValue(v);
					let link = v && v.length ?  v[0].data.LINK : 0;
					me.setVariant(link);
				},
				getTextCfg: function () {
					let res = this.callParent();
					if (!me.variant) {
						res.emptyText = me.gksd('tempVariantName');
					}
					return res;
				},
			}),
			{text: 'ОК', handler: me.submit, scope: this},
			{text: 'Отмена', handler: me.close, scope: this},
			{text: 'Применить', handler: me.apply, scope: this}
		];
	},

	initComponent: function () {
		let me = this;
		me.beforeInitComponent();
		
		let login = Ext.util.Cookies.get('Login'),
			date = Ext.Date.format(new Date(), 'd.m.Y_H:i:s');
		me.sksd('tempVariantName', login + "_" + date);
		
		me.buttons = me.createButtons();
		me.items = [
			me.feedPanel = me.createGroupPanel(),
			me.ParamsPanel = Ext.create('Ext.panel.Panel', {
				region: 'center',
				autoScroll: true,
				border: 0,
				userCls: 'rks-panel-border-bottom',
				frame: true,
				layout: {type: 'vbox', align: 'stretch'},
				itemMap: {},
				flex: 1,
				bodyPadding: 10
			})
		];
		me.callParent(arguments);
		if (me.variant) {
			me.on('afterrender', me.setVariant.bind(me, me.variant, true));			
		}
	},

	listeners: {
		beforeclose: function () {
			let me = this;
			if (!me.checkChanges) return true;

			let newData = me.dataCollector();
			if (me.oldData !== newData) {
				Ext.Msg.show({
					title: wmc.get('Attention'),
					msg: wmc.get('SaveChanges'),
					buttons: Ext.MessageBox.YESNOCANCEL,
					fn: function (buttonId) {
						if (buttonId === 'yes') {
							me.submit();
						}
						if (buttonId === 'no') {
							me.checkChanges = false;
							me.close();
						}
					},
					icon: Ext.MessageBox.QUESTION
				});
				return false;
			}
			return true;
		}
	},

	/**
	 * Загрузить вариант
	 * @param link {int} линк вариант
	 * @param fromInit {boolean} задать значение для контрола варианта
	 * */
	setVariant: function (link, fromInit) {
		let me = this;

		me.variant = link;
		if (link === 0){
			me.variant = null;
			me.dictVariant.fieldName.setValue(me.gksd('tempVariantName'));
			me.setRangCalcs([]);
			return;
		}

		const loadMask = new Ext.LoadMask({
			msg: KS.L10n.RiskInfoListView_RangCalcReading,
			target: me,
			autoShow: true,
			rid: ajaxRequest({
				url: '/SRiskInfo/LoadVariant_A',
				params: {variant: link},
				success: function (result) {
					loadMask.destroy();
					if (!result || !result.calcs) {
						console.warn('Не удалось получить данные по варианту ' + link);
						me.variant = null;
						me.dictVariant.fieldName.setValue(me.gksd('tempVariantName'));
						return false;
					}
					console.log(result.calcs);
					if (fromInit) {
						me.dictVariant.setValue(result.variant);
					}
					me.setRangCalcs(result.calcs);
					me.oldData = me.dataCollector();
				},
				failure: function (response) {
					loadMask.destroy();
					Ext.MessageBox.alert("Error", response.responseText);
				}
			}),
		});
	},

	/**
	 * Создать дерево групп критериев
	 * */
	createGroupPanel: function () {
		var me = this;
		return Ext.create('Ext.panel.Panel', {
			region: 'west',
			split: true,
			width: 200,
			border: 0,
			userCls: 'rks-panel-border-bottom',
			layout: 'fit',
			items: [
				me.groupPanel = Ext.create('Ext.tree.Panel', {
					hideHeaders: true,
					autoScroll: true,
					border: 0,
					flex: 1,
					layout: 'fit',
					tbar:
						[
							me.addBtn = Ext.create('Ext.Button',
								{
									tooltip: 'Выбрать',
									tooltipType: 'title',
									iconCls: 'x_btn_add_element',
									handler: function () {
										me.activityIndexTree = Ext.create('Keysystems.SActivityIndex.Tree',
											{
												title: 'Ранжирование',
												closeAction: 'hide',
												checkedLinks: me.rangCalcs.map(r => r.Key),
												ok: function (rows) {
													let oldLinks = me.rangCalcs.map(c => c.Key),
														newLinks = rows.map(r => r.LINK);
													if (ArrayLib.equal(oldLinks, newLinks)) return;

													let calcs = [];
													rows.forEach(row => {
														let calc = {
															Key: row.LINK,
															Caption: row.CODE + '. ' + row.NAME,
															Group: row.CGROUP,
															Type: row.TYPE,
															List: row.LIST,
															SysKey: row.TEMP_SYS_CALC
														};
														let currentCalc = me.rangCalcs.filter(r => r.Key === row.LINK)[0];
														if (currentCalc) {
															calc.Params = currentCalc.Params;
														}
														calcs.push(calc);
													});
													me.setRangCalcs(calcs);
												}
											});
										me.activityIndexTree.show();
									}
								}),
							me.deleteBtn = Ext.create('Ext.Button',
								{
									tooltip: 'Удалить',
									tooltipType: 'title',
									iconCls: 'x_btn_delete_element',
									handler: function () {
										let leaf = me.groupPanel.getSelectionModel().getSelection()[0];
										if (!leaf) return false;

										me.rangCalcs = me.rangCalcs.filter(c => c.Key != leaf.data.LINK);

										//удаляем показатель либо полностью группу (если других показателей в группе не осталось)
										if (leaf.parentNode.childNodes.length > 1) {
											leaf.parentNode.removeChild(leaf, true);
										} else {
											let group = leaf.parentNode;
											group.parentNode.removeChild(group, true);
										}
									}
								}),
						],
					caseSensitive: false,
					columns: [{
						xtype: 'treecolumn',
						text: 'Name',
						dataIndex: 'NAME',
						flex: 1,
						renderer: (value, rec) => {
							if (rec.record.data.GROUP) {
								return '<b>' + value + '</b>'
							}
							return value;
						}
					}],
					store: Ext.create('Ext.data.TreeStore', {
						fields: ['NAME', 'LINK'],
						root: {expanded: true, children: []},
						defaultRootText: ''
					}),
					rootVisible: false,
					listeners: {
						select: function (th, rec) {
							let link = rec.get('LINK'),
								calc = me.rangCalcs.filter(calc => calc.Key === link)[0];
							if (calc) {
								let type = rec.get('TYPE');
								me.ParamsPanel.itemMap['RISK_RANGES_NUM'].setVisible(type === 0);
								me.ParamsPanel.itemMap['RISK_RANGES_SUM'].setVisible(type === 1 || type === 3);
								me.ParamsPanel.itemMap['RISK_RANGES_BOOL'].setVisible(type === 2);

								for (let name in me.ParamsPanel.itemMap) {
									if (name.indexOf('RISK_RANGES') !== -1) continue;
									let fsLink = parseInt(name.split('_')[0]),
										fieldSet = me.ParamsPanel.itemMap[name];
									!link || link !== fsLink ? fieldSet.hide() : fieldSet.show();
								}

								me.setValuesInProgress = true;
								me.setParamValues(calc);
								me.setValuesInProgress = false;
							} else {
								for (let name in me.ParamsPanel.itemMap) {
									let fieldSet = me.ParamsPanel.itemMap[name];
									fieldSet.hide();
								}
							}

						}
					}
				})
			]
		});
	},

	/**
	 * Сгенерировать элементы
	 * @param calcs {Array} коллекция параметров ранжирования
	 * */
	generateElements: function (calcs) {
		let me = this,
			data = [],
			itemMap = me.ParamsPanel.itemMap;

		Ext.each(calcs, function (calc) {
			let nodeGroup = ArrayLib.filter(data, ['GROUP'], calc.Group)[0];
			if (!nodeGroup) {
				let title = "Риски по критерию " + (calc.Group === 1 ? "\"Вероятность\"" : "\"Существенность\"");
				nodeGroup = {NAME: title, GROUP: calc.Group, children: []};
				nodeGroup.leaf = false;
				nodeGroup.expanded = true;
				nodeGroup.iconCls = 'no-icon';
				data.push(nodeGroup);
			}

			let node = {NAME: calc.Caption, LINK: calc.Key, TYPE: calc.Type};
			node.leaf = true;
			node.iconCls = 'no-icon';
			nodeGroup.children.push(node);

			//#region параметры системных показателей            

			if (calc.SysKey && window.systemCalcs && !itemMap[calc.Key + '_PARAMS']) {
				let sysCalc = window.systemCalcs.filter(sysCalc => sysCalc.Key === calc.SysKey)[0];
				if (sysCalc) {
					let paramsCalc = [];
					sysCalc.Params.forEach(sysParam => {
						let controlType = miscTypes.ControlType[sysParam.ControlType],
							controlName = calc.Key + "_" + sysParam.Key;
						switch (controlType) {
							case "NumericBox":
								paramsCalc.push(me.sksc(controlName, Ext.create('Ext.form.field.Number', {
									key: controlName,
									fieldLabel: sysParam.Caption,
									labelWidth: me.labelWidth,
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								})));
								break;
							case "CurrencyBox":
								paramsCalc.push(me.sksc(controlName, Ext.create('Keysystems.Controls.CalcField', {
									key: controlName,
									fieldLabel: sysParam.Caption,
									labelWidth: me.labelWidth,
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								})));
								break;
							case "DateBox":
								paramsCalc.push(me.sksc(controlName, Ext.create('Ext.form.field.Date', {
									key: controlName,
									fieldLabel: sysParam.Caption,
									labelWidth: me.labelWidth,
									maxWidth: 150 + me.labelWidth,
									format: 'd.m.Y',
									minValue: '01.01.1900',
									maxValue: '31.12.2100',
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								})));
								break;
							case "DatePeriod":
								paramsCalc.push(me.sksc(controlName, Ext.create('Keysystems.Controls.PeriodEdit',
									{
										key: controlName,
										fieldLabel: sysParam.Caption,
										labelWidth: me.labelWidth,
										maxWidth: 450,
										listeners: {
											ksChange: function () {
												let dt = this.getValue();
												me.onChange.call(me, this, dt, true);
											},
										}
									})));
								break;
							case "CheckBox":
								paramsCalc.push(me.sksc(controlName, Ext.create('Ext.form.field.Checkbox', {
									key: controlName,
									fieldLabel: sysParam.Caption,
									labelWidth: me.labelWidth,
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								})));
								break;
							case "EditBox":
								paramsCalc.push(me.sksc(controlName, Ext.create('Ext.form.field.Text', {
									key: controlName,
									fieldLabel: sysParam.Caption,
									labelWidth: me.labelWidth,
									maxWidth: 150 + me.labelWidth,
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								})));
								break;
							case "Dictionary":
								paramsCalc.push(me.sksc(controlName,
									Ext.create('Keysystems.Controls.Dict.Edit', {
										key: controlName,
										fieldLabel: sysParam.Caption,
										labelWidth: me.labelWidth,
										mode: 'MULTI',
										listeners: {
											change: function (th, val) {
												me.onChange.call(me, th, val, true);
											},
										},
										handler: function () {
											dictFunc({
													code: sysParam.Tag,
													mode: 'MULTI',
													checkList: me.gksc(controlName).getValue(),
													parentView: me,
													control: me.gksc(controlName),
													contextSearch: me.gksc(controlName).contextSearch
												},
												{
													ok: function (v) {
														let dictEditor = me.gksc(controlName);
														if (dictEditor) {
															if (!dictEditor.key) dictEditor.key = controlName;
															dictEditor.setValue(v);
															me.onChange.call(me, dictEditor, v, true);
														}
													}
												});
										}
									})
								));
								break;
							case "ComboBox":
								paramsCalc.push(me.sksc(controlName, Ext.create('Keysystems.Controls.ComboBoxExtra', {
									key: controlName,
									fieldLabel: sysParam.Caption,
									labelWidth: me.labelWidth,
									store: CBDataLib.get(_, 0, sysParam.Tag),
									queryMode: 'local',
									displayField: 'value',
									valueField: 'index',
									value: 0,
									maxWidth: 150 + me.labelWidth,
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								})));
								break;
						}
					});

					itemMap[calc.Key + '_PARAMS'] = me.ParamsPanel.insert(0, Ext.create('Ext.form.FieldSet', {
						title: 'Параметры',
						collapsible: true,
						layout: {type: 'vbox', align: 'stretch'},
						flex: 1,
						hidden: true,
						items: paramsCalc
					}));
				}
			}

			//#endregion параметры системных показателей

			//#region параметры показателей типа Список
			if (calc.Type === 4 && calc.List && !itemMap[calc.Key + '_LIST']) {
				let paramList = [
					Ext.create('Ext.form.FieldContainer', {
						layout: {type: 'hbox', align: 'stretch'},
						pack: 'end',
						items: [
							'->',
							Ext.create('Ext.form.Label', {
								text: 'Баллы',
								margin: '0 30 0 0',
							})]
					})
				];
				let i = 0;
				calc.List.split('\r\n').forEach(value => {
					let controlName = Ext.String.format("{0}_{1}RISK_BALL_", calc.Key, i);
					paramList.push(Ext.create('Ext.form.FieldContainer', {
						layout: 'hbox',
						items: [
							Ext.create('Ext.form.field.Text', {
								fieldLabel: null,
								value: value,
								readOnly: true,
								flex: 1
							}),
							me.sksc(controlName, Ext.create('Ext.form.field.Number',
								{
									key: controlName,
									fieldLabel: null,
									padding: '0 0 0 10',
									minValue: 0,
									maxWidth: 100,
									listeners: {
										change: function (th, val) {
											me.onChange.call(me, th, val, true);
										},
									}
								}))
						]
					}));
					i++;
				});

				itemMap[calc.Key + '_LIST'] = me.ParamsPanel.insert(0, Ext.create('Ext.form.FieldSet', {
					title: 'Диапазоны групп риска',
					collapsible: true,
					layout: {type: 'vbox', align: 'stretch'},
					flex: 1,
					hidden: true,
					items: paramList
				}));
			}
			//#endregion
		});

		//#region параметры диапазонов риска
		let rangeTitle = 'Диапазоны групп риска';

		//количество, процент
		if (!itemMap['RISK_RANGES_NUM']) {
			itemMap['RISK_RANGES_NUM'] = me.ParamsPanel.add(Ext.create('Ext.form.FieldSet', {
				title: rangeTitle,
				collapsible: true,
				layout: {type: 'vbox', align: 'stretch'},
				flex: 1,
				hidden: true,
				items: [
					me.sksc('LABEL_NumRange', Ext.create('Ext.form.Label',
						{
							text: 'Баллы',
							margin: '0 0 5 410',
						})),
					me.generateNumRangeParams('Min', 'N'),
					me.generateNumRangeParams('Mid', 'N'),
					me.generateNumRangeParams('Max', 'N')
				]
			}));
		}

		//сумма
		if (!itemMap['RISK_RANGES_SUM']) {
			itemMap['RISK_RANGES_SUM'] = me.ParamsPanel.add(Ext.create('Ext.form.FieldSet', {
				title: rangeTitle,
				collapsible: true,
				layout: {type: 'vbox', align: 'stretch'},
				flex: 1,
				hidden: true,
				items: [
					me.sksc('LABEL_NumRange', Ext.create('Ext.form.Label',
						{
							text: 'Баллы',
							margin: '0 0 5 410',
						})),
					me.generateNumRangeParams('Min', 'S'),
					me.generateNumRangeParams('Mid', 'S'),
					me.generateNumRangeParams('Max', 'S')
				]
			}));
		}

		//флаг
		if (!itemMap['RISK_RANGES_BOOL']) {
			itemMap['RISK_RANGES_BOOL'] = me.ParamsPanel.add(Ext.create('Ext.form.FieldSet', {
				title: rangeTitle,
				collapsible: true,
				layout: {type: 'vbox', align: 'stretch'},
				flex: 1,
				hidden: true,
				items: [
					Ext.create('Ext.form.Label', {
						text: 'Баллы',
						margin: '0 0 5 50'
					}),
					Ext.create('Ext.form.FieldContainer', {
						layout: 'hbox',
						items: [
							Ext.create('Ext.form.field.Checkbox',
								{
									fieldLabel: null,
									checked: false,
									readOnly: true,
								}),
							me.sksc('MinBall', Ext.create('Ext.form.field.Number',
								{
									key: 'MinBall',
									fieldLabel: null,
									padding: '0 0 0 5',
									minValue: 0,
									maxWidth: 100,
									listeners: {change: (th, val) => me.onChange.call(me, th, val)}
								}))
						]
					}),
					Ext.create('Ext.form.FieldContainer', {
						layout: 'hbox',
						items: [
							Ext.create('Ext.form.field.Checkbox',
								{
									fieldLabel: null,
									checked: true,
									readOnly: true,
								}),
							me.sksc('MaxBall', Ext.create('Ext.form.field.Number',
								{
									key: 'MaxBall',
									fieldLabel: null,
									padding: '0 0 0 5',
									minValue: 0,
									maxWidth: 100,
									listeners: {change: (th, val) => me.onChange.call(me, th, val)}
								}))
						]
					})
				]
			}));
		}

		//#endregion параметры диапазонов риска

		return data;
	},

	/**
	 * Событие изменения значения контрола
	 * @param th {object} элемент
	 * @param value {object} значение
	 * @param fullKey {boolean} составной ключ элемента
	 * */
	onChange: function (th, value, fullKey) {
		let me = this;

		if (me.setValuesInProgress) return;
		let node = me.groupPanel.getSelectionModel().getSelection()[0];
		if (!node) return;
		let link = node.data.LINK,
			paramKey = fullKey ? th.key : link + "_" + th.key,
			calc = me.rangCalcs.filter(calc => calc.Key === link)[0],
			param = calc.Params.filter(param => param.Key === paramKey)[0];

		if (param) {
			let val = value;
			switch (param.ControlType) {
				case 3: //DateBox
					val = value.toDateString();
					break;
				case 6:
					val = value && value.length ? value.map(v => {
						return {LINK: v.data.LINK, CODE: v.data.CODE, NAME: v.data.NAME}
					}) : [];
					break;
			}
			param.Tag = val;
			console.log('Изменение значения параметра ' + param.Key);
			console.log(value);
		}
	},

	/**
	 * Сгенерировать числовые параметры диапазонов риска
	 * @param what {string} префикс параметра [Min|Mid|Max]
	 * @param type {string} тип показателя, для которого генерируются параметры [N|S]
	 * */
	generateNumRangeParams: function (what, type) {
		let me = this,
			label = (what === 'Min' ? 'Минимальный' : (what === 'Mid' ? 'Средний' : 'Максимальный')) + ' риск',
			controlType = type === 'N' ? 'Ext.form.field.Number' : 'Keysystems.Controls.CalcField';

		return Ext.create('Ext.form.FieldContainer', {
			layout: 'hbox',
			items: [
				me.sksc(Ext.String.format('{0}{1}1', what, type), Ext.create(controlType,
					{
						key: Ext.String.format('{0}{1}1', what, type),
						fieldLabel: label,
						labelWidth: 130,
						minValue: 0,
						maxWidth: 230,
						listeners: {change: (th, val) => me.onChange.call(me, th, val)}
					})),
				me.sksc(Ext.String.format('{0}{1}2', what, type), Ext.create(controlType,
					{
						key: Ext.String.format('{0}{1}2', what, type),
						fieldLabel: '...',
						labelSeparator: '',
						labelWidth: 12,
						padding: '0 0 0 5',
						minValue: 0,
						maxWidth: 110,
						listeners: {change: (th, val) => me.onChange.call(me, th, val)}
					})),
				me.sksc(Ext.String.format('{0}{1}Ball', what, type), Ext.create('Ext.form.field.Number',
					{
						key: Ext.String.format('{0}{1}Ball', what, type),
						fieldLabel: null,
						padding: '0 0 0 30',
						minValue: 0,
						maxWidth: 100,
						listeners: {change: (th, val) => me.onChange.call(me, th, val)}

					}))
			]
		});
	},

	/**
	 * Инициализировать коллекцию параметров ранжирования, сгенерировать контролы по ним
	 * @param calcs {Array} коллекция параметров ранжирования
	 * */
	setRangCalcs: function (calcs) {
		let me = this;
		me.rangCalcs = calcs;

		let checkCalc = function (calc, name) {
			let paramKey = calc.Key + "_" + name,
				param = calc.Params.filter(param => param.Key === paramKey)[0];
			if (!param) calc.Params.push({
				Key: paramKey
			});
		};
		calcs.forEach(calc => {
			if (!calc.Params) calc.Params = [];
			switch (calc.Type) {
				case 0: //Количество
					checkCalc(calc, 'MinN1');
					checkCalc(calc, 'MinN2');
					checkCalc(calc, 'MidN1');
					checkCalc(calc, 'MidN2');
					checkCalc(calc, 'MaxN1');
					checkCalc(calc, 'MaxN2');

					checkCalc(calc, 'MinNBall');
					checkCalc(calc, 'MidNBall');
					checkCalc(calc, 'MaxNBall');
					break;
				case 1://Сумма
				case 3://Процент
					checkCalc(calc, 'MinS1');
					checkCalc(calc, 'MinS2');
					checkCalc(calc, 'MidS1');
					checkCalc(calc, 'MidS2');
					checkCalc(calc, 'MaxS1');
					checkCalc(calc, 'MaxS2');
					checkCalc(calc, 'MinSBall');
					checkCalc(calc, 'MidSBall');
					checkCalc(calc, 'MaxSBall');
					break;
				case 2:
					checkCalc(calc, 'MinBall');
					checkCalc(calc, 'MaxBall');
					break;
				case 4:
					let i = 0;
					calc.List.split('\r\n').forEach(value => {
						let paramKey = Ext.String.format('{0}_{1}RISK_BALL_', calc.Key, i),
							param = calc.Params.filter(param => param.Key === paramKey)[0];
						if (!param) {
							let calcParam = {
								Key: paramKey,
								ControlType: 5 //EditBox
							};
							calc.Params.push(calcParam);
						}
						i++;
					});
					break;
			}

			if (calc.SysKey && window.systemCalcs) {
				let sysCalc = window.systemCalcs.filter(sysCalc => sysCalc.Key === calc.SysKey)[0];
				if (sysCalc) {
					sysCalc.Params.forEach(sysParam => {
						let paramKey = calc.Key + "_" + sysParam.Key;
						let param = calc.Params.filter(param => param.Key === paramKey)[0];
						if (!param) {
							let calcParam = {
								Key: paramKey,
								ControlType: sysParam.ControlType
							};
							calc.Params.push(calcParam);
						}
					});
				}
			}


		});
		let groups = me.generateElements(me.rangCalcs);
		me.groupPanel.store.setRootNode({expanded: true, children: groups});
	},

	/**
	 * Задать значения контролов для параметров показателя ранжирования
	 * @param calc {object} показатель ранжирования
	 * */
	setParamValues: function (calc) {
		try {
			let me = this;

			let setValue = function (controlName, paramName) {
				if (!paramName) paramName = calc.Key + "_" + controlName;
				let param = calc.Params && calc.Params.length ? calc.Params.filter(par => par.Key === paramName)[0] : null,
					value = param ? param.Tag : null,
					control = me.gksc(controlName);

				if (!param) {
					console.warn('Не найден параметр показателя [Key=' + calc.Key + '] с ключом ' + paramName);
					return;
				}

				if (!control) {
					console.warn('Не найден элемент с ключом ' + controlName);
					return;
				}

				let controlType = miscTypes.ControlType[param.ControlType] ?? 0;
				switch (controlType) {
					case "DatePeriod":
						value ? control.setValue(value.dh1, value.dh2) : control.setValue(longPeriod.begin, longPeriod.end);
						break;
					default:
						control.setValue(value);
				}
				if (value) {
					console.log('Установка значения для ' + controlName);
					console.log(value);
				}
			}

			let setValue2 = function (name) {
				setValue(name, name);
			}

			if (calc.Type === 0) {
				setValue('MinN1');
				setValue('MinN2');
				setValue('MidN1');
				setValue('MidN2');
				setValue('MaxN1');
				setValue('MaxN2');

				setValue('MinNBall');
				setValue('MidNBall');
				setValue('MaxNBall');

			} else if (calc.Type === 1 || calc.Type === 3) {
				setValue('MinS1');
				setValue('MinS2');
				setValue('MidS1');
				setValue('MidS2');
				setValue('MaxS1');
				setValue('MaxS2');
				setValue('MinSBall');
				setValue('MidSBall');
				setValue('MaxSBall');
			} else if (calc.Type === 2) {
				setValue('MinBall');
				setValue('MaxBall');
			} else if (calc.Type === 4) {
				let i = 0;
				calc.List.split('\r\n').forEach(value => {
					let controlName = Ext.String.format("{0}_{1}RISK_BALL_", calc.Key, i);
					setValue2(controlName);
					i++;
				});
			}

			if (calc.SysKey && window.systemCalcs) {
				let sysCalc = window.systemCalcs.filter(sysCalc => sysCalc.Key === calc.SysKey)[0];
				if (sysCalc) {
					sysCalc.Params.forEach(param => {
						setValue2(calc.Key + "_" + param.Key);
					});
				}
			}
		} catch (ex) {
			console.error(ex);
		}
	},

	dataCollector: function () {
		return JSON.stringify(this.rangCalcs);
	}
});