﻿/**
 * A {@link Ext.ux.statusbar.StatusBar} plugin that provides automatic error
 * notification when the associated form contains validation errors.
 */
Ext.define('Keysystems.Controls.ErrorStatus', {
	extend: 'Ext.Component',
	requires: ['Ext.util.MixedCollection', 'Ext.ux.StatusBar'],
	alias: 'widget.ErrorStatus',
	/**
	* @cfg {String} errorIconCls
	* The {@link Ext.ux.statusbar.StatusBar#iconCls iconCls} value to be applied
	* to the status message when there is a validation error.
	*/
	errorIconCls: 'x-status-error',
	/**
	* @cfg {String} errorListCls
	* The css class to be used for the error list when there are validation errors.
	*/
	errorListCls: 'x-status-error-list',
	/**
	* @cfg {String} validIconCls
	* The {@link Ext.ux.statusbar.StatusBar#iconCls iconCls} value to be applied
	* to the status message when the form validates.
	*/
	validIconCls: 'x-status-valid',

	/**
	* @cfg {String} showText
	* The {@link Ext.ux.statusbar.StatusBar#text text} value to be applied when
	* there is a form validation error.
	*/
	showText: '',
	/**
	* @cfg {String} hideText
	* The {@link Ext.ux.statusbar.StatusBar#text text} value to display when
	* the error list is displayed.
	*/
	hideText: 'Скрыть?',
	/**
	* @cfg {String} submitText
	* The {@link Ext.ux.statusbar.StatusBar#text text} value to be applied when
	* the form is being submitted.
	*/
	submitText: 'Saving...',

	// private
	init: function (sb) {
		var me = this;

		me.statusBar = sb;
		sb.on({
			single: true,
			scope: me,
			render: me.onStatusbarRender,
			beforedestroy: me.destroy
		});
		sb.on({
			click: {
				element: 'el',
				fn: me.onStatusClick,
				scope: me,
				buffer: 200
			}
		});
	},

	onStatusbarRender: function (sb) {
		var me = this,
            startMonitor = function () {
            	me.monitor = true;
            };

		me.monitor = true;
		me.errors = Ext.create('Ext.util.MixedCollection');
		me.listAlign = (sb.statusAlign === 'right' ? 'br-tr?' : 'bl-tl?');

		if (me.form) {
			//me.formPanel = Ext.getCmp(me.form);
			//me.basicForm = me.formPanel.getForm();
			me.formPanel = me.form;
			me.basicForm = me.form;
			me.startMonitoring();

			me.basicForm.on('beforeaction', function (f, action) {
				if (action.type === 'submit') {
					// Ignore monitoring while submitting otherwise the field validation
					// events cause the status message to reset too early
					me.monitor = false;
				}
			});

		}
	},

	// private
	startMonitoring: function () {
		this.basicForm.on('errormessage', this.onFieldValidation, this);
	},

	// private
	stopMonitoring: function () {
		this.basicForm.un('errormessage', this.onFieldValidation, this);
	},

	// private
	onDestroy: function () {
		this.stopMonitoring();
		this.statusBar.statusEl.un('click', this.onStatusClick, this);
		this.callParent(arguments);
	},

	// private
	onFieldValidation: function (caption, message) {
		var me = this;

		if (!me.monitor) {
			return false;
		}

		me.errors.removeAll();
		if (message!='')
		me.errors.add(caption, { caption: caption, msg: message });

		this.updateErrorList();

		if (me.errors.getCount() > 0) {
			if (me.statusBar.getText() !== caption) {
				if (caption == '') {
					me.statusBar.clearStatus().setIcon(me.validIconCls);
					me.progressBar.show();
				} else {
					me.progressBar.hide();
					me.showText = caption;
					me.statusBar.setStatus({
						text: me.showText,
						iconCls: me.errorIconCls
					});
				}
			}
		} else {
			me.statusBar.clearStatus().setIcon(me.validIconCls);
		}
	},

	// private
	updateErrorList: function () {
		var me = this,
            msg,
            msgEl = me.getMsgEl();

		if (me.errors.getCount() > 0) {
			msg = ['<ul>'];
			this.errors.each(function (err) {
				msg.push('<li id="x-err-', err.caption, '"><a href="#">', err.msg, '</a></li>');
			});
			msg.push('</ul>');
			msgEl.update(msg.join(''));
		} else {
			msgEl.update('');
		}
		// reset msgEl size
		msgEl.setSize('auto', 'auto');
	},

	// private
	getMsgEl: function () {
		var me = this,
            msgEl = me.msgEl,
            t;

		if (!msgEl) {
			msgEl = me.msgEl = Ext.DomHelper.append(me.basicForm.body, {
				cls: me.errorListCls
			}, true);
			msgEl.hide();
			msgEl.on('click', function (e) {
				t = e.getTarget('li', 10, true);
				if (t) {
				    if (t.id.split('x-err-') != null && t.id.split('x-err-')[1] != null) {
				        var cmp = Ext.getCmp(t.id.split('x-err-')[1]);
				        if (cmp!=null)cmp.focus();
				    }
				    me.hideErrors();
				}
			}, null, { stopEvent: true }); // prevent anchor click navigation
		}
		return msgEl;
	},

	// private
	showErrors: function () {
		var me = this;

		me.updateErrorList();
		me.getMsgEl().alignTo(me.statusBar.getEl(), me.listAlign).slideIn('b', { duration: 100, easing: 'easeOut' });
		me.statusBar.setText(me.hideText);
		me.formPanel.body.on('click', me.hideErrors, me, { single: true }); // hide if the user clicks directly into the form
	},

	// private
	hideErrors: function () {
		var me = this;
		var el = me.getMsgEl();
		if (el.isVisible()) {
			el.slideOut('b', { duration: 100, easing: 'easeIn' });
			if (this.statusBar.getText() != this.showText)
				this.statusBar.setText(this.showText);

		}
		this.formPanel.body.un('click', this.hideErrors, this);
		
		
	},

	// private
	onStatusClick: function () {
		if (this.getMsgEl().isVisible()) {
			this.hideErrors();
		} else if (this.errors.getCount() > 0) {
			this.showErrors();
		}
	}
});