import os
from datetime import datetime
import subprocess
from pydub import AudioSegment
from config import AUDIO


def make_audio_dir(sub_dir_name: str, datetime_now: datetime):
    path_to_dir = os.path.os.path.join(
        os.getcwd(), f'audio-files/{sub_dir_name}-{datetime_now.strftime("%d-%m-%Y")}')

    if not os.path.exists(path_to_dir):
        os.makedirs(path_to_dir)

    return path_to_dir


def save_audio_file(file_bytes: bytes, directory_path: str, datetime_now: datetime):
    file_path = os.path.os.path.join(
        directory_path, f'audio-{datetime_now.strftime("%H-%M-%S-%f")[:-3]}.wav')

    wav_bytes = convert_to_wav(file_bytes=file_bytes)

    with open(file_path, "wb") as f:
        f.write(wav_bytes)

    return file_path


def convert_to_wav(file_bytes: bytes):

    process = subprocess.Popen(
        [
            "ffmpeg",
            "-i", "pipe:0",
            "-f", "wav",
            "-acodec", "pcm_s16le",
            "-ar", "44100",
            "-ac", "2",
            "pipe:1"
        ],
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )

    wav_bytes, stderr = process.communicate(input=file_bytes)

    return wav_bytes


def split_audio(file_path: str, directory_path: str, datetime_now: datetime):
    audio = AudioSegment.from_file(file_path, format="wav")

    total_length = len(audio)

    chunks_paths = []

    if total_length < AUDIO.chunk_max_size:
        return None

    for i in range(0, total_length, AUDIO.chunk_max_size):
        chunk = audio[i:i + AUDIO.chunk_max_size]

        chunk_name = os.path.join(
            directory_path, f'chunk_{i // AUDIO.chunk_max_size:03d}_{datetime_now.strftime("%H-%M-%S-%f")[:-3]}.wav')

        chunk.export(chunk_name, format="wav")

        chunks_paths.append(chunk_name)

    return chunks_paths
