from fastapi.responses import JSONResponse
from fastapi import APIRouter, Request, Depends, UploadFile, File
from sqlalchemy.ext.asyncio import AsyncSession
from datetime import datetime

from services.core.database.database_service import get_async_session
from services.core.system.system_utils import check_auth, get_system_token_from_cookie

from services.core.variables import DEFAULT_EXCEPTION_RESPONSE

from config import AUDIO

from features.record.record_utils import save_audio_file, make_audio_dir, split_audio
from ml.model_utils import transcribe
from services.core.authorization.authorization_service import get_access_token_from_cookie
from services.core.system.system_db_operations import get_system_id
from features.record.record_db_operations import add_record_db
from preload.globals import MODEL

from features.record.record_schemas import Record


router = APIRouter(
    prefix="/record",
    tags=["Record"],
    default_response_class=JSONResponse
)


@router.post("/predict_text", summary="Predict Text (Предсказать текс)", responses={
    200: {
        "description": "При успешном возвращает status = success и предсказанный текст",
        "content": {
            "application/json": {
                "example":
                    {
                        "status": "succes", "message": "Запрос успешно выполнен.", "data": {"predicted_text": "Текст, который предсказала модель"}
                    },
            }
        }
    },
    400: {
        "description": "Возвращает json со статусом и описанием ошибки",
        "content": {
            "application/json": {
                "example": DEFAULT_EXCEPTION_RESPONSE
            }
        }
    }
})
@check_auth(is_system_token_check=True, is_user_token_check=True, hard_auth_check=False)
async def predict_text(request: Request,
                       audio: UploadFile = File(...),
                       session: AsyncSession = Depends(get_async_session)):
    """  
    Присутствует ограничение размера файла и размер чанка аудио из настроек.

    Тело запроса:
    - **audio**:File(Required) - webm файл с аудио
    """
    contents = await audio.read()

    if len(contents) > AUDIO.audiofile_max_size:
        return JSONResponse(content={"status": "failed", "message": f"Файл превышает размер {AUDIO.audiofile_max_size_mb}МБ"})

    datetime_now = datetime.now()

    records_dir = make_audio_dir(
        sub_dir_name="records", datetime_now=datetime_now)

    audio_path = save_audio_file(
        file_bytes=contents, directory_path=records_dir, datetime_now=datetime_now)

    chunks_dir = make_audio_dir(
        sub_dir_name="chunks", datetime_now=datetime_now)

    audio_chunks = split_audio(file_path=audio_path, directory_path=chunks_dir,
                               datetime_now=datetime_now)
    text = ""

    if audio_chunks:
        for chunk in audio_chunks:
            text += transcribe(AUDIO.is_openai_model, MODEL, chunk) + " "
    else:
        text = transcribe(AUDIO.is_openai_model, MODEL, audio_path)

    text = text.strip().capitalize()

    system_token = get_system_token_from_cookie(request)
    access_token = get_access_token_from_cookie(request)

    system_id = 1

    if access_token == None:
        system_id = await get_system_id(system_token, session)

    record_item = Record(date=datetime_now, system_id=system_id,
                         predicted_text=text, model_name=AUDIO.model_name)

    await add_record_db(record_item, session)

    return JSONResponse(content={
        "status": "success",
        "message": "Запрос успешно выполнен.",
        "data": {"predicted_text": text}
    })
