﻿Ext.define('Svod.ColumnSortingManager', {
    alias: 'column.sortingManager',
    singleton: true,

    addColumnSorter: function (grid, toolbar) {
        var me = this;
        if (!grid)
            return;

        // Добавляем кнопку
        if (toolbar) {
            // в переданный тулбар
            toolbar.add({
                xtype: 'button',
                iconCls: 'ks-icon-table_view',
                tooltip: 'Настройка сортировки',
                handler: function ()  {
                    me.createSortManagerWindow(grid);
                }
            });
        }
        else {
            // в колонку с нумерацией
            var columns = grid.getVisibleColumns();
            if (columns.length === 0 || columns[0].xtype !== 'rownumberer')
                return;

            columns[0].setText(
                '<span style="cursor:pointer;" data-qtip="Настройка сортировки">' +
                '<img src="PlatformHandler.axd?icon=table_view.png"/></span>');
            columns[0].on('headerclick', me.createSortManagerWindow, this, { args: [grid] });
        }
    },

    createSortManagerWindow: function(gridForSorting) {
        var me = this,
            sortOptionsTreeGrid = me.createSortOptionsTreeGrid(gridForSorting);

        var sortManagerWindow = new Ext.Window({
            title: 'Настройка сортировки',
            modal: true,
            plain: true,
            width: 400,
            height: 600,
            items: sortOptionsTreeGrid,
            layout: 'fit',
            buttonAlign: 'right',
            buttons: [
                {
                    text: 'ОК',
                    cls: 'marked-button',
                    handler: function () {
                        me.updateSorting(gridForSorting);
                        sortManagerWindow.close();
                    }
                },
                {
                    text: 'Отмена',
                    cls: 'dim-button',
                    handler: function () {
                        sortManagerWindow.close();
                    }
                }
            ]
        });
        sortManagerWindow.show();
    },

    createSortOptionsTreeGrid: function (gridForSorting) {
        var me = this,
            sortOptionsData = me.getSortOptionsData(gridForSorting);

        var columns = me.getSortOptionsColumns();
        if (me.hasGroups)
            columns[0].hidden = false;

        me.sortOptionsTreeGrid = new Ext.tree.Panel({
            width: '100%',
            height: '100%',
            sortableColumns: false,
            columnLines: true,
            rowLines: true,
            allowDeselect: false,
            rootVisible: false,
            multiSelect: false,
            useArrows: true,
            columns,
            store: Ext.create('Ext.data.TreeStore', {
                data: {
                    children: sortOptionsData
                }
            }),
            updateActiveRec: me.updateActiveRec,
            hasGroups: me.hasGroups,
            viewConfig: {
                plugins: {
                    treeviewdragdrop: {
                        dragText: 'Переместите строку для изменения порядка сортировки'
                    }
                }
            },
            listeners: {
                cellclick: me.onCellClickHandler,
                beforedrop: me.onBeforeDropHandler,
                drop: me.onDropHandler
            }
        });

        return me.sortOptionsTreeGrid;
    },

    getSortOptionsData: function (gridForSorting) {
        var me = this,
            sorters = {};

        // Получаем информацию о примененной сортировке
        gridForSorting.getStore().getSorters().each(function (sorter) {
            sorters[sorter.getId()] = sorter.getDirection();
        });

        me.hasGroups = false;

        var sortOptionsData = [];
        var tempGroup = {};
        gridForSorting.eachColumnCfg(function (column) {
            if (column.visibility && column.width > 0 && !KS.isEmpty(column.dataIndex) && column.dataIndex !== 'CHECKED') {
                var data = {
                    dataIndex: column.dataIndex,
                    columnName: column.text,
                    sortDirection: sorters[column.dataIndex]
                        ? sorters[column.dataIndex]
                        : null,
                    leaf: true
                }

                if (column.parentGroup) {
                    // Необходимо добавить данные в группу
                    if (tempGroup.columnName !== column.parentGroup.header) {
                        if (!Ext.Object.isEmpty(tempGroup)) {
                            sortOptionsData.push(tempGroup);
                        }

                        // Создаем родительский узел дерева
                        tempGroup = {
                            columnName: column.parentGroup.header,
                            sortDirection: null,
                            expanded: true,
                            children: []
                        };
                    }

                    me.hasGroups = true;
                    tempGroup.children.push(data);
                }
                else {
                    sortOptionsData.push(data);
                }
            }
        });

        if (!Ext.Object.isEmpty(tempGroup)) {
            sortOptionsData.push(tempGroup);
        }

        return sortOptionsData;
    },

    getSortOptionsColumns: function () {
        return [
            {
                xtype: 'treecolumn',
                menuDisabled: true,
                resizable: false,
                width: 22,
                hidden: true
            },
            {
                menuDisabled: true,
                resizable: false,
                width: 32
            },
            {
                text: 'Колонка',
                dataIndex: 'columnName',
                menuDisabled: true,
                flex: 1
            },
            {
                text: '<img src="PlatformHandler.axd?icon=sort_updown.png"/>',
                dataIndex: 'sortDirection',
                tooltip: 'Направление сортировки',
                menuDisabled: true,
                resizable: false,
                width: 32,
                renderer: function (value, metadata) {
                    metadata.tdStyle = 'background-position: center !important;';

                    switch (value) {
                        case 'Ascending':
                        case 'ASC':
                            metadata.css += ' arrow_top';
                            break;

                        case 'Descending':
                        case 'DESC':
                            metadata.css += ' arrow_bottom';
                            break;

                        default:
                            metadata.tdStyle += ' background-image: none !important;'
                            break;
                    }
                    return '';
                }
            }
        ];
    },

    onCellClickHandler: function (tableView, td, cellIndex, record, tr, rowIndex, e) {
        tableView.grid.updateActiveRec(record);

        if (e.position.column.dataIndex === 'sortDirection') {
            switch (record.data.sortDirection) {
                case 'Ascending':
                case 'ASC':
                    record.set('sortDirection', 'DESC');
                    break;

                case 'Descending':
                case 'DESC':
                    record.set('sortDirection', null);
                    break;

                default:
                    record.set('sortDirection', 'ASC');
                    break;
            }

            // Если есть childNodes, то выставялем им такой же sortDirection
            Ext.Array.each(record.childNodes, function (childRec) {
                childRec.set('sortDirection', record.data.sortDirection);
            });
        }
    },

    onBeforeDropHandler: function (node, data, overModel, dropPosition, dropHandlers) {
        if (data.records[0].parentNode !== overModel.parentNode ||
            (overModel.parentNode.id === 'root' && dropPosition === 'append')) {
            // Запрещаем перенос элементов одной группы в другую, а также перенос всей группы внутрь другой группы
            dropHandlers.cancelDrop();

            data.view.grid.updateActiveRec(data.records[0]);
        }
    },

    onDropHandler: function (node, data) {
        data.view.grid.updateActiveRec(data.records[0]);
    },

    updateActiveRec: function (record) {
        var view = this.getView(),
            store = this.getStore(),
            rowIndex = store.indexOf(record),
            colIndex = this.hasGroups ? 1 : 0,
            oldActiveRec = this.activeRec;

        if (oldActiveRec) {
            // Убираем иконку перемещения
            var oldRecRowIndex = store.indexOf(oldActiveRec);
            var oldRecCell = view.getCellByPosition({ row: oldRecRowIndex, column: colIndex });
            Ext.fly(oldRecCell).removeCls('ks-icon-move_updown');
        }

        // Устанавливаем иконку перемещения активному record
        var recCell = view.getCellByPosition({ row: rowIndex, column: colIndex });
        Ext.fly(recCell).addCls('ks-icon-move_updown');

        this.activeRec = record;
    },

    updateSorting: function (gridForSorting) {
        var sorters = [];
        this.sortOptionsTreeGrid.getStore().each(function (rec) {
            if (rec.data.sortDirection && rec.data.children === null) {
                sorters.push({
                    property: rec.data.dataIndex,
                    direction: rec.data.sortDirection
                });
            }
        });

        gridForSorting.suspendEvents();

        // Устанавливаем новую сортировку
        gridForSorting.lastSort = {};
        var store = gridForSorting.getStore();
        store.getSorters().removeAll();
        store.setSorters(sorters);

        gridForSorting.resumeEvents();

        // GetControlData
        store.loadPageFromServerTemplate(1);
    }
});