#!/bin/bash

###
### WEB-Сервер-КС
###
### Copyright (c) 2022, ООО "Кейсистемс"
###

# Обновление каталога dotnet приложения

# Определяем каталог выполнения текущего скрипта
dks_bashfile_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

while [[ "$#" -gt 0 ]]
  do
    case $1 in
      -t|--app_type)
        app_type="$2"
        ;;

      -a|--app_tar_archive)
        app_tar_archive="$2"
        ;;

      -p|--kestrel_port)
        kestrel_port="$2"
        ;;

      -s|--inet_source)
        inet_source="$2"
        ;;
      -f|--force)
        force="1"
        ;;

      --help)
        show_help="1"
        ;;
    esac
    shift
done

if [ ! -z "${show_help}" ]; then

echo -e "
 параметры                 по умолчанию   описание
 -------------------       ------------   ------------------------------------------------------------------------------------
 -t   | --app_type                        Тип обновляемого приложения.
                                          Параметр поддерживает указание запускающего модуля, если приложение
                                          нужно запустить нестандартно. Для этого необходимо после типа указать двоеточие
                                          и имя стартового модуля, например: \"wBKS:web_core.dll\".

 -a   | --app_tar_archive                 Имя архива *.tar.gz для развертывания, обязательное условие - наличие одной.
                                          корневой папки внутри архива.

 -p   | --kestrel_port     54430          Кестрел порт, назначенный приложению. Используется для всех типов приложений.

 -s   | --inet_source                     Интернет-ресурс для закачки файла архива. Закачка осуществляется через утилиту wget.

 -f   | --force                           Принудительное обновление, даже если файл архива не менялся на интернет-ресурсе

"
exit 1;
fi;

if [ -z "${kestrel_port}" ]; then
  kestrel_port="54430"
fi;


# Путь установки
dks_inst_path=$($dks_bashfile_dir/ws_const_var.sh dks_inst_path)
dks_cont_name=$($dks_bashfile_dir/ws_const_var.sh dks_cont_name)
host_stdout=$($dks_bashfile_dir/ws_const_var.sh host_stdout)
cont_stdout=$($dks_bashfile_dir/ws_const_var.sh cont_stdout)
dks_serv_name="${dks_cont_name}.service"


# Цвета
ErrColor=$($dks_bashfile_dir/ws_const_var.sh ErrColor)
SuccColor=$($dks_bashfile_dir/ws_const_var.sh SuccColor)
WarnColor=$($dks_bashfile_dir/ws_const_var.sh WarnColor)
NoColor=$($dks_bashfile_dir/ws_const_var.sh NoColor)

sOK=$(echo $($dks_bashfile_dir/ws_const_var.sh sOK) | tr '#' ' ')
wOK=$(echo $($dks_bashfile_dir/ws_const_var.sh wOK) | tr '#' ' ')
eRR=$(echo $($dks_bashfile_dir/ws_const_var.sh eRR) | tr '#' ' ')
emptE=$(echo $($dks_bashfile_dir/ws_const_var.sh emptE) | tr '#' ' ')
emptEsh=$($dks_bashfile_dir/ws_const_var.sh emptE)

function fn_file_clr(){
cat > $1 << EOF
EOF
}


function fn_file_echo() {
local LeftSpace=$2
LeftSpace=$(echo ${LeftSpace} | tr '#' ' ')
cat $1 | while read line
do
 echo -e "${LeftSpace}$line"
done
}


# Находим приложение и виртуальный каталог
wskstype="${dks_inst_path}/www/html/wsks_${kestrel_port}/wskstype"

if [ -f ${wskstype} ]; then
  app_type=$(echo $(cat ${wskstype} | grep "type=") | sed 's|.*=||')
  app_virt=$(echo $(cat ${wskstype} | grep "virt_catalog=") | sed 's|.*=||')
  app_note=$(echo $(cat ${wskstype} | grep "note") | sed 's|.*=||')
  app_type_name=$($dks_bashfile_dir/ws_getntype.sh ${app_type})
fi;


echo ""
echo "           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
echo "           ~   Обновление прикладного приложения WEB-Сервер-КС     "
echo "           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
echo "           ~                                                       "
echo "           ~    Порт кестрел : ${kestrel_port}"
if [ -f ${wskstype} ]; then
echo "           ~    Тип          : ${app_type} - ${app_type_name}"
echo "           ~    Вирт.каталог : ${app_virt}"
if ! [ -z "${app_note}" ]; then
echo "           ~    Примечание   : ${app_note}"
fi;
fi;
echo "           ~                                                       "
echo "           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"


# Обработка параметра типа с переданной запускающей dll
# формат передачи <тип>:<имя dll>
app_type_all=${app_type}
app_startdll=""
if [[ "${app_type_all}" == *:* ]]; then
  app_type=$(echo ${app_type_all} | sed -r 's/:.+//')
  app_startdll=$(echo ${app_type_all} | sed 's|.*:||')
else
  app_type=${app_type_all}
fi;


# app_type
bash ${dks_bashfile_dir}/ws_checkparams.sh "app_type" "${app_type}"
if [ $? = "1" ]; then exit 1; fi;


# Проверка существования каталога приложения
target_app_catalog="${dks_inst_path}/www/html/wsks_${kestrel_port}"
if ! [ -d "${target_app_catalog}/" ]; then
  echo -e "${eRR}Проверка существования каталога обновляемого приложения."
  echo -e "${emptE}Отсутствует каталог приложения."
  echo -e "${emptE}Возможно приложение не устанавливалось или неправильно указан порт."
  echo ""
  exit 1;
else
  echo -e "${sOK}Проверка существования каталога обновляемого приложения."
fi;


# Если есть источник для закачки, то пробуем закачать
if ! [ -z "${inet_source}" ]; then
  bash ${dks_bashfile_dir}/ws_wget.sh ${inet_source} ${app_tar_archive}
  ret_exit="$?"
  if [ "${ret_exit}" == "1" ]; then
    # ошибка закачки файлов
    echo ""
    exit 1;
  fi;

  if [ "${ret_exit}" == "2" ]; then
    # отсутствие новых файлов для обновления
    # если не задан принудительный режим, то выходим
    if [ -z "${force}" ]; then
      echo ""
      exit 0;
    else
      echo -e "${sOK}Включен режим принудительного обновления."
    fi;
  fi;
fi;

# Проверка архива приложения
target_app_tar_file="${dks_bashfile_dir}/tar_files/${app_tar_archive}"

if [ ! -f "${target_app_tar_file}" ]; then
  echo -e "${eRR}Не найден файл архива обновляемого приложения."
  echo -e "${emptE}Файл архива необходимо поместить в каталог - ${dks_bashfile_dir}/tar_files/."
  echo -e ""
  exit 1;
fi;

# Проверка на соответствие типа обновляемого приложения
wskstype="${target_app_catalog}/wskstype"
app_type_upd=$(echo $(cat ${wskstype} | grep "type=") | sed 's|.*=||')

if [ "${app_type_upd}" != "${app_type}" ] ; then
  echo -e "${eRR}Проверка совпадения типа обновляемого приложения и архива обновления."
  echo -e "${emptE}Тип существующего приложения \"${app_type_upd}\" и тип архива обновления \"${app_type}\", указанный в параметрах не совпадают."
  echo -e "${emptE}Проверьте правильность указания параметров и в правильности использования архива обновления."
  echo -e ""
  exit 1;
fi;


# Определяем запущено ли приложение
app_status=$(${dks_bashfile_dir}/ws_info_app.sh "${kestrel_port}" "get_status")

if [ "${app_status}" = "Run" ]; then
  # останавливаем только приложение, если запущено
  bash ${dks_bashfile_dir}/ws_ctlapp.sh -p "${kestrel_port}" -m "stop"
fi;

# создаем временный каталог обновления
target_temp_catalog="${dks_bashfile_dir}/tmp/___tmp_ksws_update_${kestrel_port}"
mkdir -p ${target_temp_catalog}

echo -e "${sOK}Создание временного каталога ${target_temp_catalog}."


# распаковываем туда архив
tar zxf ${target_app_tar_file} -C ${target_temp_catalog} --strip-components 1
if [ $? = "0" ]; then
  echo -e "${sOK}Распаковка архива обновления \"${app_tar_archive}\" во временный каталог."
else
  # удаляем временный каталог обновления
  rm -rf ${target_temp_catalog}
  echo -e "${eRR}Ошибка распаковки архива обновления \"${app_tar_archive}\" во временный каталог."
  echo -e "${emptE}Удаление временного каталога."
  echo -e "${emptE}Операция обновления будет прервана."
  echo -e ""
  exit 1;
fi;


# \App_Data\ - не обновлять
# \wwwroot\App_data\ - не обновлять
# appsettings.Production.json - не обновлять

# во временном каталоге удаляем каталоги, которые не обновляем
# + файлы настроек и конфигураций и копируем их с обновляемого
# приложения, установленного в ${target_app_catalog}

appSetProd="appsettings.Production.json"
appLoginInfo="logininfo.txt"
appUploadConf="uploadservice.config"
appUpdateConf="updateservice.config"
AppDir1="App_data"
AppDir2="wwwroot/App_data"
AppDir3_wDWH="Service"
AppDir4_sID="cert"

# файл appsettings.Production.json
if [ -f "${target_app_catalog}/${appSetProd}" ]; then
  if [ -f "${target_temp_catalog}/${appSetProd}" ]; then
    rm "${target_temp_catalog}/${appSetProd}"
  fi;
  cp ${target_app_catalog}/${appSetProd}   ${target_temp_catalog}/${appSetProd}
fi;

# файл logininfo.txt - файл уведомлений
if [ -f "${target_app_catalog}/${appLoginInfo}" ]; then
  if [ -f "${target_temp_catalog}/${appLoginInfo}" ]; then
    rm "${target_temp_catalog}/${appLoginInfo}"
  fi;
  cp ${target_app_catalog}/${appLoginInfo}   ${target_temp_catalog}/${appLoginInfo}
fi;


# файл uploadservice.config

if [ -f "${target_app_catalog}/${appUploadConf}" ]; then
  if [ -f "${target_temp_catalog}/${appUploadConf}" ]; then
    rm "${target_temp_catalog}/${appUploadConf}"
  fi;
  cp ${target_app_catalog}/${appUploadConf}   ${target_temp_catalog}/${appUploadConf}
fi;

# файл updateservice.config
if [ -f "${target_app_catalog}/${appUpdateConf}" ]; then
  if [ -f "${target_temp_catalog}/${appUpdateConf}" ]; then
    rm "${target_temp_catalog}/${appUpdateConf}"
  fi;
  cp ${target_app_catalog}/${appUpdateConf}   ${target_temp_catalog}/${appUpdateConf}
fi;

# каталог App_data
if [ -d "${target_app_catalog}/${AppDir1}" ]; then
  if [ -d "${target_temp_catalog}/${AppDir1}" ]; then
    rm -R "${target_temp_catalog}/${AppDir1}"
  fi;
  cp -r ${target_app_catalog}/${AppDir1} ${target_temp_catalog}
fi;

# каталог wwwroot/App_data
if [ -d "${target_app_catalog}/${AppDir2}" ]; then
  if [ -d "${target_temp_catalog}/${AppDir2}" ]; then
    rm -R "${target_temp_catalog}/${AppDir2}"
  fi;

  mkdir -p ${target_temp_catalog}/${AppDir2}
  cp -r ${target_app_catalog}/${AppDir2}/. ${target_temp_catalog}/${AppDir2}

fi;

# каталог Service
if [ "${app_type}" = "wDWH" ]; then
  if [ -d "${target_app_catalog}/${AppDir3_wDWH}" ]; then
    if [ -d "${target_temp_catalog}/${AppDir3_wDWH}" ]; then
      rm -R "${target_temp_catalog}/${AppDir3_wDWH}"
    fi;
    cp -r ${target_app_catalog}/${AppDir3_wDWH} ${target_temp_catalog}
  fi;
fi;

# каталог cert
if [ "${app_type}" = "sID" ]; then
  if [ -d "${target_app_catalog}/${AppDir4_sID}" ]; then
    if [ -d "${target_temp_catalog}/${AppDir4_sID}" ]; then
      rm -R "${target_temp_catalog}/${AppDir4_sID}"
    fi;
    cp -r ${target_app_catalog}/${AppDir4_sID} ${target_temp_catalog}
  fi;
fi;


# переносим с заменой в рабочий каталог приложения
echo -e "${sOK}Сохранение файлов конфигураций и служебных подкаталогов."

# полностью очищаем рабочий каталог приложения
rm -rf ${target_app_catalog}/*
echo -e "${sOK}Удаление содержимого рабочего каталога."

# копируем файлы и каталоги из временного в рабочий
cp -r ${target_temp_catalog}/* ${target_app_catalog}
echo -e "${sOK}Копирование данных в рабочий каталог из временного."

# считаем хеш файла архива
archivehash=$(sha1sum "${target_app_tar_file}" | awk '{ print $1 }')
# определяем версию приложения
app_file_ver=$(${dks_bashfile_dir}/ws_getverfname.sh "${app_type}")
app_version=$(${dks_bashfile_dir}/ws_getver.sh "${target_app_catalog}/${app_file_ver}")


# создаем файл-информатор обновляемого приложения
echo "type=${app_type}"             > ${target_app_catalog}/wskstype
app_type_name=$(${dks_bashfile_dir}/ws_getntype.sh ${app_type})
echo "type_name=${app_type_name}"   >> ${target_app_catalog}/wskstype
echo "kestrel_port=${kestrel_port}" >> ${target_app_catalog}/wskstype
echo "virt_catalog=${app_virt}"     >> ${target_app_catalog}/wskstype
echo "note=${app_note}"             >> ${target_app_catalog}/wskstype

echo "archive=${app_tar_archive}"   >> ${target_app_catalog}/wskstype
echo "archivehash=${archivehash}"   >> ${target_app_catalog}/wskstype
echo "version=${app_version}"       >> ${target_app_catalog}/wskstype


# права доступа и владелец, второй параметр исключаемые к установке прав файлы (через пробелы)
bash ${dks_bashfile_dir}/ws_setright.sh "${target_app_catalog}" "${appSetProd}" "${target_temp_catalog}"
echo -e "${sOK}Коррекция прав доступа."

# удаляем временный каталог обновления
rm -rf ${target_temp_catalog}
echo -e "${sOK}Удаление временного каталога."


if [ "${app_status}" = "Run" ]; then
  # Запускаем приложение, если на момент обновления было запущено
  bash ${dks_bashfile_dir}/ws_ctlapp.sh -p "${kestrel_port}" -m "start"
fi;

# Запуск info
bash ${dks_bashfile_dir}/ws_info_app.sh "${kestrel_port}"

echo ""

exit 0;



