#!/bin/bash

###
### WEB-Сервер-КС
###
### Copyright (c) 2023, ООО "Кейсистемс"
###

ws_host=$1
ws_port=$2
ws_type=$3
prometheus_port=$4
contsoft=$5
locale=$6
rebuild=$7

typeOS="ksdeb12"
no_ssl=""

# === Определяем каталог выполнения текущего скрипта ==========================
dks_bashfile_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

if [ "${ws_host}" = "127.0.0.1" ]; then
  ip_host=$(${dks_bashfile_dir}/sys/sys_hostip.sh)
  if ! [ -z "${ip_host}" ]; then
    ws_host="${ip_host}"
  fi;
fi;

if [ -z "${ws_port}" ] ; then
  ws_port="443"
fi;

if [ -z "${ws_type}" ] || [[ "${ws_type}" != "apache2" && "${ws_type}" != "angie" && "${ws_type}" != "no_http" ]]; then
  ws_type="nginx"
fi;


if [ -z "${no_ssl}" ] ; then
  ssl_mode="1"
else
  ssl_mode="0"
fi;

tag_name=$($dks_bashfile_dir/dks_const_var.sh         tag_name)
dks_cont_name=$($dks_bashfile_dir/dks_const_var.sh    dks_cont_name)
dks_inst_path=$($dks_bashfile_dir/dks_const_var.sh    dks_inst_path)
app_inst_path=$($dks_bashfile_dir/dks_const_var.sh    app_inst_path)
app_var_lib_path=$($dks_bashfile_dir/dks_const_var.sh app_var_lib_path)

dks_stdout=$($dks_bashfile_dir/dks_const_var.sh dks_stdout)


# версия
ws_ver=$(${dks_bashfile_dir}/dks_const_var.sh ws_ver)
dks_version=$(sed -n 1,1p ${dks_bashfile_dir}/dks_version)


# Цвета
sOK=$(echo $($dks_bashfile_dir/dks_const_var.sh sOK) | tr '#' ' ')
wOK=$(echo $($dks_bashfile_dir/dks_const_var.sh wOK) | tr '#' ' ')
eRR=$(echo $($dks_bashfile_dir/dks_const_var.sh eRR) | tr '#' ' ')
emptE=$(echo $($dks_bashfile_dir/dks_const_var.sh emptE) | tr '#' ' ')
emptEsh=$($dks_bashfile_dir/dks_const_var.sh emptE)

# sudo
is_root=$($dks_bashfile_dir/dks_const_var.sh is_root)

function fn_file_clr(){
cat > $1 << EOF
EOF
}

function fn_file_echo() {
local LeftSpace=$2
LeftSpace=$(echo ${LeftSpace} | tr '#' ' ')
cat $1 | while read line
do
 echo -e "${LeftSpace}$line"
 done
}


# ========= Блок проверок параметров =============================
echo ""
wsks_msg_title="WEB-Сервер-КС"

# Проверка соответствия веб сервера и базового образа
if [ "${ws_type}" = "angie" ] && [ "${typeOS}" = "ksdeb09" ]; then

  echo -e "${eRR}WEB сервер типа ${ws_type} не поддерживается для образа контейнера ${typeOS}."
  echo -e "${emptE}Операция создания контейнера ${dks_cont_name} невозможна."
  echo ""
  exit 1;

fi;


# Проверка прав суперпользователя
bash ${dks_bashfile_dir}/sys/sys_is_root.sh "install"
if [ $? = "1" ]; then exit 1; fi;

# Проверка что контейнер не запущен в режиме перестройки
if ! [ -z "${rebuild}" ] ; then

  dks_cont_name_check="${dks_cont_name}_${ws_port}"

  # Проверяем существование контейнера в docker/podman
  dks_exists=$(${contsoft} container ls -a | grep ${dks_cont_name_check})
  if [ -z "$dks_exists" ]; then
    echo -e "${eRR}Проверка существования контейнера ${dks_cont_name_check}."
    echo -e "${emptE}Контейнера ${dks_cont_name} не существует. Операция пересоздания невозможна."
    echo ""
    exit 1;
  fi;

  dks_status=$(${contsoft} inspect -f '{{.State.Running}}' ${dks_cont_name_check})
  if [ "${dks_status}" = "true" ]; then

    echo -e "${eRR}Проверка работы контейнера ${dks_cont_name_check}."
    echo -e "${emptE}Внимание!!! Для пересоздания контейнер ${dks_cont_name_check} должен быть остановлен."
    echo -e "${emptE}Произведите остановку с помощью команды: systemctl stop ${dks_cont_name_check}"
    echo -e "${emptE}Операция пересоздания невозможна."
    echo ""
    exit 1;

  fi;
fi;


# Проверка не пустого порта в диапазоне [5444-5493]
port_correct="no"
if [ "${ws_port}" = "443" ]; then
  port_correct="yes"
else
  bash ${dks_bashfile_dir}/sys/sys_checkport.sh "interval" "${ws_port}" "${wsks_msg_title}" "5444" "5493"
  if [ $? = "1" ]; then exit 1; fi;
  port_correct="yes"
fi;

# Проверка "свободности" порта на хосте (только при установке с http сервером)
if ! [ "${ws_type}" = "no_http"  ]; then
  bash ${dks_bashfile_dir}/sys/sys_checkport.sh "free" "${ws_port}" "${wsks_msg_title}"
  if [ $? = "1" ]; then exit 1; fi;
fi;

# Проверка "свободности" порта prometheus ${prometheus_port} на хосте
bash ${dks_bashfile_dir}/sys/sys_checkport.sh "free" "${prometheus_port}" "prometheus"
if [ $? = "1" ]; then exit 1; fi;


# Проверка сервиса docker или файла podman на хосте
if [ "${contsoft}" = "docker"  ]; then
  contsoftname="Docker"
  consoftchecktype="service"
else
  contsoftname="Podman"
  consoftchecktype="file"
fi;

bash ${dks_bashfile_dir}/sys/sys_checkservice.sh "${contsoft}" "${contsoftname}" "${consoftchecktype}"
if [ $? = "1" ]; then exit 1; fi;

#  Проверка поддерживаемых локалей
isLCOK="no"
aLC_support=("be_BY.UTF-8" "en_US.UTF-8" "es_CU.UTF-8" "ru_RU.UTF-8" "tg_TJ.UTF-8" "ru_RU.utf8")

# Проверка докали из массива доступных
case "${aLC_support[@]}" in  *"${locale}"*) isLCOK="yes" ;; esac

if [ "${isLCOK}" = "no" ] ; then
  echo -e "${eRR}Указана неподдерживаемая локаль \"${locale}\" для ${wsks_msg_title}."
  echo -e "${emptE}Значение локали должно быть: ${aLC_support[@]}."
  echo -e "${emptE}Операция установки будет прервана."
  echo -e ""
  exit 1;
fi;




# ========= Завершение блока проверок параметров =====================

# Окончательный путь к развертыванию
inst_dir="${app_inst_path}_${ws_port}"
dks_cont_name="${dks_cont_name}_${ws_port}"
dks_serv_name="${dks_cont_name}.service"


# Проверка существования каталога, если есть - отменяем установку
if [ -d "${inst_dir}/" ]; then

  if [ -z "${rebuild}" ] ; then

    echo -e "${eRR}Проверка существования структуры каталогов для запуска контейнера ${dks_cont_name}."
    echo -e "${emptE}Внимание!!! Уже существует структура каталогов для работы контейнера ${dks_cont_name}."
    echo -e "${emptE}Если Вам необходимо полностью пересоздать структуру каталогов, предварительно"
    echo -e "${emptE}произведите полное удаление с помощью команды: ./uninstall.sh -p ${ws_port} -all,"
    echo -e "${emptE}и запустите создание контейнера заново."
    echo -e "${emptE}ПРИ ПОЛНОМ УДАЛЕНИИ ОБЯЗАТЕЛЬНО УБЕДИТЕСЬ, ЧТО ДАННЫЕ В КАТАЛОГЕ ${inst_dir}/www/html ВАМ ТОЧНО НЕ НУЖНЫ..."
    echo -e "${emptE}Операция создания контейнера ${dks_cont_name} невозможна."
    echo ""
    exit 1;

  fi;

fi;


se_mode="Disabled"
# Проверка Selinux
if [ -f /etc/selinux/config ]; then
  se_mode=$(echo $(getenforce))
  if [ "${se_mode}" = "Enforcing" ]; then
    # переключаем в разрешительный :)
    setenforce Permissive
  fi;
fi;


# Удаление принудительно контейнер+образ #############################
${contsoft} container inspect ${dks_cont_name} >/dev/null 2>&1
if [ $? -eq 0 ]; then

  echo -e "${sOK}Удаление контейнера ${dks_cont_name}."
  ${contsoft} rm -f ${dks_cont_name} >/dev/null

fi;

${contsoft} image inspect ${dks_cont_name} >/dev/null 2>&1
if [ $? -eq 0 ]; then

  echo -e "${sOK}Удаление образа ${dks_cont_name}."
  ${contsoft} rmi -f ${dks_cont_name} >/dev/null

fi;

${contsoft} image inspect ${tag_name} >/dev/null 2>&1
if [ $? -eq 0 ]; then
  echo -e "${sOK}Удаление эталонного локального образа ${tag_name}."
  ${contsoft} rmi -f ${tag_name} >/dev/null
fi;
####################################################################

# === Загружаем образ ==================================================================

dks_image_tar_file="${typeOS}_${tag_name}_image.tar"
if [ ! -f "${dks_inst_path}/container_image/${dks_image_tar_file}" ]; then
  echo -e "${eRR}Загрузка эталонного локального образа ${tag_name} из архива ${dks_image_tar_file}"
  echo -e "${emptE}Отсутствует файл архива с образом ${dks_inst_path}/container_image/${dks_image_tar_file}"
  echo -e "${emptE}Операция создания контейнера ${dks_cont_name} невозможна."

  # восстанавливаем значение для selinux
  if [ "${se_mode}" = "Enforcing" ]; then
    # переключаем обратно.
    setenforce Enforcing
  fi;

  exit 1;
else
  ${contsoft} load -q -i "${dks_inst_path}/container_image/${dks_image_tar_file}" >/dev/null
  echo -e "${sOK}Загрузка эталонного локального образа ${tag_name} из архива ${dks_image_tar_file}."
fi

# Временная зона хоста для синхронизации даты и времени в контейнере
tzhost=$(timedatectl status | grep "Time zone:" | sed -e 's/^[ ]*Time zone: \(.*\) (.*)$/\1/g')
tzhost=$(echo ${tzhost} | sed 's/\//\\\//g')

# === Создаем конечный образ на базе загруженного =====================================

# Формируем файл Dockerfile из шаблона
cp ${dks_inst_path}/container_image/Dockerfile.t ${dks_inst_path}/container_image/Dockerfile
sed -i 's/@ws_port@/'${ws_port}'/g'              ${dks_inst_path}/container_image/Dockerfile
sed -i 's/@locale@/'${locale}'/g'                ${dks_inst_path}/container_image/Dockerfile
sed -i 's/@tzhost@/'${tzhost}'/g'                ${dks_inst_path}/container_image/Dockerfile


${contsoft} build -t ${dks_cont_name} ${dks_inst_path}/container_image >/dev/null
echo -e "${sOK}Построение конечного образа ${dks_cont_name} на базе эталонного."

# Удаляем Dockerfile
rm -f ${dks_inst_path}/container_image/Dockerfile

# === Создаем структуру каталогов, если нет ===========================================
# переходим в корневой каталог и проверяем наличие структуры

cd /
if ! [ -d "${inst_dir}/" ]; then
  echo -e "${emptE}Структура каталогов ${inst_dir} для запуска контейнера ${dks_cont_name} не существует, создаем..."
  mkdir -p ${inst_dir}/conf.ks
  mkdir -p ${inst_dir}/conf.ks/vhost.ks

  # копируем файлы конфига http сервера
  if [ "${ws_type}" = "nginx" ]; then
    cp ${dks_bashfile_dir}/wsks_files/conf.ks/nginx.conf   ${inst_dir}/conf.ks/nginx.conf
  fi;
  if [ "${ws_type}" = "angie" ]; then
    cp ${dks_bashfile_dir}/wsks_files/conf.ks/angie.conf   ${inst_dir}/conf.ks/angie.conf
  fi;
  if [ "${ws_type}" = "apache2" ]; then
    # копируем файлы конфига апача
    cp ${dks_bashfile_dir}/wsks_files/conf.ks/apache2.conf ${inst_dir}/conf.ks/apache2.conf
  fi;

  # каталоги запуска и управления web приложениями
  mkdir -p ${inst_dir}/ctl/sys
  mkdir -p ${inst_dir}/ctl/app
  mkdir -p ${inst_dir}/ctl/tar_files

  # системные файлы
  cp ${dks_bashfile_dir}/sys/sys_exec_cont.sh               ${inst_dir}/ctl/sys/sys_exec_cont.sh
  cp ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh            ${inst_dir}/ctl/sys/sys_exec_cont_ex.sh
  cp ${dks_bashfile_dir}/sys/sys_checkport.sh               ${inst_dir}/ctl/sys/sys_checkport.sh
  cp ${dks_bashfile_dir}/sys/sys_const_var.sh               ${inst_dir}/ctl/sys/sys_const_var.sh
  cp ${dks_bashfile_dir}/sys/sys_checkproc.sh               ${inst_dir}/ctl/sys/sys_checkproc.sh
  cp ${dks_bashfile_dir}/sys/sys_hostip.sh                  ${inst_dir}/ctl/sys/sys_hostip.sh

  # управляющие файлы
  cp $(find ${dks_bashfile_dir}/wsks_files/ctl/ -name "app_*.sh") ${inst_dir}/ctl/
  cp $(find ${dks_bashfile_dir}/wsks_files/ctl/ -name "ws_*.sh")  ${inst_dir}/ctl/


  # Формируем через шаблон -> @ws_port@ @ws_ver@
  cp ${dks_bashfile_dir}/wsks_files/ctl/ws_const_var.sh.t   ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@ws_ver@/'${ws_ver}'/g'                         ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@ws_host@/'${ws_host}'/g'                       ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@ws_port@/'${ws_port}'/g'                       ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@ws_type@/'${ws_type}'/g'                       ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@contsoft@/'${contsoft}'/g'                     ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@locale@/'${locale}'/g'                         ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@prometheus_port@/'${prometheus_port}'/g'       ${inst_dir}/ctl/ws_const_var.sh
  sed -i 's/@dks_version@/'"${dks_version}"'/g'             ${inst_dir}/ctl/ws_const_var.sh


  # Формируем через шаблон -> @ws_ver@
  cp ${dks_bashfile_dir}/wsks_files/ctl/ws_start.sh.t       ${inst_dir}/ctl/ws_start.sh
  sed -i 's/@ws_ver@/'${ws_ver}'/g'                         ${inst_dir}/ctl/ws_start.sh

  # файл readme.txt
  cp ${dks_bashfile_dir}/wsks_files/template/readme.txt.t   ${inst_dir}/ctl/readme.txt
  sed -i 's/@ws_ver@/'${ws_ver}'/g'                         ${inst_dir}/ctl/readme.txt
  # доформировываем @ws_showhelp@
  bash ${inst_dir}/ctl/ws_showhelp.sh "${inst_dir}/ctl/readme.txt"

  chmod -R 777 ${inst_dir}/ctl
  chmod 644 ${inst_dir}/ctl/readme.txt

  mkdir -p ${inst_dir}/log
  mkdir -p ${inst_dir}/ssl

  if [ "${ssl_mode}" = "1" ]; then

    if [ "${ws_type}" = "nginx" ]; then
      cp ${dks_bashfile_dir}/wsks_files/conf.ks/nginx_sslks.conf    ${inst_dir}/conf.ks/sslks.conf
    fi;

    if [ "${ws_type}" = "angie" ]; then
      cp ${dks_bashfile_dir}/wsks_files/conf.ks/angie_sslks.conf    ${inst_dir}/conf.ks/sslks.conf
    fi;

    if [ "${ws_type}" = "apache2" ]; then
      cp ${dks_bashfile_dir}/wsks_files/conf.ks/apache2_sslks.conf  ${inst_dir}/conf.ks/sslks.conf
    fi;


    ## скопировать файлы *.crt и *.key
    ## дать права на файл *.key
    cp ${dks_bashfile_dir}/wsks_files/ssl/wsks-19.08_demo.key ${inst_dir}/ssl/wsks-19.08_demo.key
    cp ${dks_bashfile_dir}/wsks_files/ssl/wsks-19.08_demo.crt ${inst_dir}/ssl/wsks-19.08_demo.crt
    chmod 0600 ${inst_dir}/ssl/wsks-19.08_demo.key
  fi;


  # каталог шаблонов
  mkdir -p ${inst_dir}/template
  # файл запуска
  cp ${dks_bashfile_dir}/wsks_files/template/5xxx_app.sh.t         ${inst_dir}/template/5xxx_app.sh.t

  # Формируем vhost (виртуальные хосты, он один) через шаблон (для ssl свой шаблон)
  if [ "${ws_type}" = "nginx" ]; then
    cp ${dks_bashfile_dir}/wsks_files/template/nginx_server.conf.t  ${inst_dir}/conf.ks/nginx_server.conf
    sed -i 's/@ws_port@/'${ws_port}'/g'                             ${inst_dir}/conf.ks/nginx_server.conf
  fi;

  if [ "${ws_type}" = "angie" ]; then
    cp ${dks_bashfile_dir}/wsks_files/template/angie_server.conf.t  ${inst_dir}/conf.ks/angie_server.conf
    sed -i 's/@ws_port@/'${ws_port}'/g'                             ${inst_dir}/conf.ks/angie_server.conf
  fi;

  if [ "${ws_type}" = "apache2" ]; then
    if [ "${ssl_mode}" = "1" ]; then
      cp ${dks_bashfile_dir}/wsks_files/template/apache2_vhost_ks.ssl.conf.t ${inst_dir}/conf.ks/vhost_ks.conf
      sed -i 's/@ws_port@/'${ws_port}'/g'                                    ${inst_dir}/conf.ks/vhost_ks.conf
    else
      cp ${dks_bashfile_dir}/wsks_files/template/apache2_vhost_ks.conf.t     ${inst_dir}/conf.ks/vhost_ks.conf
      sed -i 's/@ws_port@/'${ws_port}'/g'                                    ${inst_dir}/conf.ks/vhost_ks.conf
    fi;
  fi;

  # шаблон для конкретных vhost приложений и точек EndPoint SignalR
  if [ "${ws_type}" = "nginx" ]; then
    cp ${dks_bashfile_dir}/wsks_files/template/nginx_5xxx_ks.conf.t   ${inst_dir}/template/5xxx_ks.conf.t
    cp ${dks_bashfile_dir}/wsks_files/template/nginx_5xxx_vh.conf.t   ${inst_dir}/template/5xxx_vh.conf.t
  fi;

  if [ "${ws_type}" = "angie" ]; then
    cp ${dks_bashfile_dir}/wsks_files/template/angie_5xxx_ks.conf.t   ${inst_dir}/template/5xxx_ks.conf.t
    cp ${dks_bashfile_dir}/wsks_files/template/angie_5xxx_vh.conf.t   ${inst_dir}/template/5xxx_vh.conf.t
  fi;

  if [ "${ws_type}" = "apache2" ]; then
    cp ${dks_bashfile_dir}/wsks_files/template/apache2_5xxx_ks.conf.t ${inst_dir}/template/5xxx_ks.conf.t
    cp ${dks_bashfile_dir}/wsks_files/template/apache2_5xxx_vh.conf.t ${inst_dir}/template/5xxx_vh.conf.t
  fi;


  # шаблоны для конфиг файлов приложений по маске
  cp $(find ${dks_bashfile_dir}/wsks_files/template/ -name "*.Production.json.t") ${inst_dir}/template/

  if [ "${ws_type}" = "apache2" ]; then
    if [ "${ssl_mode}" = "1" ]; then
      cp ${dks_bashfile_dir}/wsks_files/template/apache2_ports_ks.ssl.conf.t     ${inst_dir}/conf.ks/ports_ks.conf
      sed -i 's/@ws_port@/'${ws_port}'/g'                                        ${inst_dir}/conf.ks/ports_ks.conf
    else
      cp ${dks_bashfile_dir}/wsks_files/template/apache2_ports_ks.conf.t         ${inst_dir}/conf.ks/ports_ks.conf
      sed -i 's/@ws_port@/'${ws_port}'/g'                                        ${inst_dir}/conf.ks/ports_ks.conf
    fi;
  fi;

  # Шаблон для prometheus
  cp ${dks_bashfile_dir}/wsks_files/template/5xxx_prometheus.json.t              ${inst_dir}/template/5xxx_prometheus.json.t

  # каталоги для размещения файлов приложений
  mkdir -p ${inst_dir}/www/html

  # файл запрета сожержимого каталогов
  cp ${dks_bashfile_dir}/wsks_files/template/index.html.t          ${inst_dir}/www/html/index.html

  # каталоги для размещения данных приложений
  mkdir -p ${app_var_lib_path}_${ws_port}

  # файл лицензии
  cp ${dks_inst_path}/COPYRIGHT ${inst_dir}/COPYRIGHT

  # Вспомогательные программы prometheus
  # cоздание каталогов для конфигов
  mkdir -p ${inst_dir}/conf.etc
  mkdir -p ${inst_dir}/conf.etc/prometheus
  mkdir -p ${inst_dir}/conf.etc/prometheus/sd_files

  # cоздание каталогов для данных
  mkdir -p ${app_var_lib_path}_${ws_port}/prometheus


  # prometheus
  cp ${dks_bashfile_dir}/wsks_files/template/prometheus/prometheus.yml.t   ${inst_dir}/conf.etc/prometheus/prometheus_${ws_port}.yml
  sed -i 's/@ws_port@/'${ws_port}'/g'                                      ${inst_dir}/conf.etc/prometheus/prometheus_${ws_port}.yml

  cp ${dks_bashfile_dir}/wsks_files/template/prometheus/prometheus_web.yml.t ${inst_dir}/conf.etc/prometheus/prometheus_web_${ws_port}.yml
  sed -i 's/@ws_port@/'${ws_port}'/g'                                        ${inst_dir}/conf.etc/prometheus/prometheus_web_${ws_port}.yml

  cp ${dks_bashfile_dir}/wsks_files/template/prometheus/prometheus_sm.sh.t ${inst_dir}/ctl/app/prometheus_sm.sh
  sed -i 's/@prometheus_port@/'${prometheus_port}'/g'                      ${inst_dir}/ctl/app/prometheus_sm.sh
  sed -i 's/@tag_name@/'${tag_name}'/g'                                    ${inst_dir}/ctl/app/prometheus_sm.sh
  sed -i 's/@ws_port@/'${ws_port}'/g'                                      ${inst_dir}/ctl/app/prometheus_sm.sh
  sed -i 's/@ws_type@/'${ws_type}'/g'                                      ${inst_dir}/ctl/app/prometheus_sm.sh


  chmod +x ${inst_dir}/ctl/app/prometheus_sm.sh

  ## скопировать файлы *.crt и *.key
  ## дать права на файл *.key
  cp ${dks_bashfile_dir}/wsks_files/ssl/wsks-19.08_demo.key ${inst_dir}/conf.etc/prometheus/wsks-19.08_demo.key
  cp ${dks_bashfile_dir}/wsks_files/ssl/wsks-19.08_demo.crt ${inst_dir}/conf.etc/prometheus/wsks-19.08_demo.crt

  chmod 644 ${inst_dir}/conf.etc/prometheus/wsks-19.08_demo.crt
  chmod 644 ${inst_dir}/conf.etc/prometheus/wsks-19.08_demo.key


  # помещаем файл информатор, если по контейнеризации podman
  rm -f ${inst_dir}/podman
  if [ "${contsoft}" = "podman" ]; then
    echo "podman" > ${inst_dir}/podman
  fi;
  # файл версии
  rm -f ${inst_dir}/dks_version
  echo "${dks_version}" > ${inst_dir}/dks_version

  echo -e "${sOK}Создание структуры каталогов для запуска контейнера ${dks_cont_name}."

else

  if [ -z "${rebuild}" ] ; then

    # резервный проверочный код
    echo -e "${eRR}Проверка существования структуры каталогов для запуска контейнера ${dks_cont_name}."
    echo -e "${emptE}Внимание!!! Уже существует структура каталогов для работы контейнера ${dks_cont_name}."
    echo -e "${emptE}Если Вам необходимо полностью пересоздать структуру каталогов, предварительно"
    echo -e "${emptE}произведите полное удаление с помощью команды: ./uninstall.sh -p ${ws_port} -all,"
    echo -e "${emptE}и запустите создание контейнера заново."
    echo -e "${emptE}ПРИ ПОЛНОМ УДАЛЕНИИ ОБЯЗАТЕЛЬНО УБЕДИТЕСЬ, ЧТО ДАННЫЕ В КАТАЛОГЕ ${inst_dir}/www/html ВАМ ТОЧНО НЕ НУЖНЫ..."
    echo -e "${emptE}Операция создания контейнера ${dks_cont_name} невозможна."

    # восстанавливаем значение для selinux
    if [ "${se_mode}" = "Enforcing" ]; then
      # переключаем обратно.
      setenforce Enforcing
    fi;

    echo ""

    exit 1;
  fi;

fi;

# === Проверка наличие модулей ЭЦП и если нет, то создание каталогов - заглушек ====================
if ! [ -d "/opt/cprocsp/" ]; then
  mkdir -p /opt/cprocsp
  mkdir -p /etc/opt/cprocsp
  mkdir -p /var/opt/cprocsp
fi;

if ! [ -d "/opt/itcs/" ]; then
  mkdir -p /opt/itcs
  mkdir -p /etc/opt/itcs
  mkdir -p /var/opt/itcs
fi;

# возвращаемся в каталог запуска установки
cd ${dks_bashfile_dir}

# === Создаем контейнер из образа и запускаем в фоновом режиме =======================================
${contsoft} run -d \
           -v ${inst_dir}/conf.ks:/etc/${ws_type}/conf.ks \
           -v ${inst_dir}/ssl:/etc/${ws_type}/ssl \
           -v ${inst_dir}/log:/var/log/${ws_type} \
           -v ${inst_dir}/www:/var/www \
           -v ${inst_dir}/ctl:/opt/${tag_name}/ctl \
           -v ${inst_dir}/conf.etc/prometheus:/etc/prometheus \
           -v /opt/cprocsp:/opt/cprocsp \
           -v /opt/itcs:/opt/itcs \
           -v /etc/opt/cprocsp:/etc/opt/cprocsp \
           -v /etc/opt/itcs:/etc/opt/itcs \
           -v /var/opt/cprocsp:/var/opt/cprocsp \
           -v /var/opt/itcs:/var/opt/itcs \
           -v ${app_var_lib_path}_${ws_port}:/var/lib/${dks_cont_name} \
           -i \
           --network host \
           --name ${dks_cont_name} ${dks_cont_name} /bin/bash >/dev/null

echo -e "${sOK}Создание контейнера ${dks_cont_name} и запуск в ${contsoft} (порт\\id ${ws_port})."

# переходим в корневой каталог и проверяем наличие структуры
cd /

# Проверка наличие модулей ЭЦП и указание путей к модулям ЭЦП
if [ -d "/opt/cprocsp/lib/amd64/" ]; then
  cont_cmd="echo \"/opt/cprocsp/lib/amd64\" > /etc/ld.so.conf.d/cpro.conf && ldconfig"

  #${contsoft} exec -d -t ${dks_cont_name} su -c "${cont_cmd}"
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh  -cs "${contsoft}"      \
                                                    -cn "${dks_cont_name}" \
                                                    -c  "${cont_cmd}"      \
                                                    -noout

fi;

if [ -d "/opt/itcs/lib/" ]; then
  cont_cmd="echo \"/opt/itcs/lib\" > /etc/ld.so.conf.d/itcs.conf && ldconfig"
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh  -cs "${contsoft}"      \
                                                    -cn "${dks_cont_name}" \
                                                    -c  "${cont_cmd}"      \
                                                    -noout

fi;

# Для контейнеров на базе D12 создан link на библиотеку libdl.so.2
if [ "${typeOS}" = "ksdeb12" ]; then
  cont_cmd="ln -s /lib/x86_64-linux-gnu/libdl.so.2 /lib/x86_64-linux-gnu/libdl.so"
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh  -cs "${contsoft}"      \
                                                    -cn "${dks_cont_name}" \
                                                    -c  "${cont_cmd}"      \
                                                    -noout

fi;


fn_file_clr ${dks_stdout}
chmod 777 ${dks_stdout}

# Создание пользователя ks-www-data на хосте

bash ${dks_bashfile_dir}/sys/sys_useradd.sh "ks-www-data"
if [ $? = "1" ]; then exit 1; fi;


# Корректируем пользователя www-data в контейнере
# Замена ID пользователя www-data на хостовый ID (считываем в переменные)
# нужно для того чтобы синхронизировать права доступа к каталогу .../www/html
# Пользователи с равными ID будут наследовать и права доступа
# Поэтому выравниваем id для пользователя www-data
# В контейнере -> /etc/passwd -> www-data:x:33:33:www-data:/var/www:/bin/bash
# символ "/" экранируем в hex-16 -> \x2F

uid_whost=$(id -u ks-www-data)
gid_whost=$(id -g ks-www-data)

# синхронизация пользователей
cont_cmd="groupmod -g ${gid_whost} www-data && usermod -u ${uid_whost} -g ${gid_whost} www-data"
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${cont_cmd}" -noout

# коррекция прав на каталоги запуска
right_cmd="chown www-data:www-data /opt/${tag_name}/ctl/app/*.sh"
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${right_cmd}" -noout

# каталоги приложений
right_cmd="chown -R www-data:www-data /var/www"
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${right_cmd}" -noout

# каталоги логов
right_cmd=" chmod 777 -R /var/log/${ws_type}"
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${right_cmd}" -noout

# каталоги для размещения данных приложений
right_cmd=" chown -R www-data:www-data /var/lib/${dks_cont_name}"
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${right_cmd}" -noout

# удаляем бинарники ненужных web серверов
del_cmd_apache="apt remove -y apache2* && if [ -d \"/etc/apache2/\" ]; then rm -R /etc/apache2; fi;"
del_cmd_angie="apt remove -y angie     && if [ -d \"/etc/angie/\" ];   then rm -R /etc/angie;   fi;"
del_cmd_nginx="apt remove -y nginx     && if [ -d \"/etc/nginx/\" ];   then rm -R /etc/nginx;   fi;"

if [ "${ws_type}" = "nginx" ]; then
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_angie}" -noout
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_apache}" -noout
fi;

if [ "${ws_type}" = "apache2" ]; then
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_nginx}" -noout
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_angie}" -noout
fi;

if [ "${ws_type}" = "angie" ]; then
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_apache}" -noout
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_nginx}" -noout
fi;

if [ "${ws_type}" = "no_http" ]; then
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_angie}" -noout
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_apache}" -noout
  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${del_cmd_nginx}" -noout
fi;

if [ -z "${ws_type}" ] || [[ "${ws_type}" != "apache2" && "${ws_type}" != "angie" &&  "${ws_type}" != "no_http" ]]; then
  ws_type="nginx"
fi;


# Останавливаем контейнер
${contsoft} stop -t 1 ${dks_cont_name} >/dev/null
echo -e "${sOK}Остановка контейнера ${dks_cont_name}."


# возвращаемся в каталог запуска установки
cd ${dks_bashfile_dir}

# === Создание автозапуска при перезагрузке хоста =======================================================

if [ -f "/lib/systemd/system/${dks_serv_name}" ]; then
  fn_file_clr /lib/systemd/system/${dks_serv_name}
fi;

# Формирование содержимого файла */service
if [ "${contsoft}" = "docker"  ]; then
  requires_service="Requires=docker.service"
  after_service="After=docker.service"
else
  requires_service=""
  after_service=""
fi;


cat > /lib/systemd/system/${dks_serv_name} << EOF
[Unit]
Description=(WSKS)${tag_name}:${ws_port} (${contsoft}/${typeOS})
# Запускать после старта сервисов и служб
${requires_service}
After=syslog.target
After=network.target
${after_service}

[Service]
Type=forking
WorkingDirectory=${inst_dir}/ctl
ExecStart=${inst_dir}/ctl/ws_start.sh
ExecStop=${inst_dir}/ctl/ws_stop.sh
RemainAfterExit=yes

# Запрет на убийство сервиса вследствие нехватки памяти и срабатывания механизма OOM:
#-1000 полный запрет (такой у sshd стоит), -100 понизим вероятность.
OOMScoreAdjust=-1000

# Таймаут в секундах, сколько ждать system отработки старт/стоп команд.
TimeoutSec=300

[Install]
#Уровень запуска - Многопользовательский режим без графики
WantedBy=multi-user.target
EOF


systemctl daemon-reload

fn_file_echo ${dks_stdout} ${emptEsh}
systemctl enable ${dks_serv_name} >/dev/null 2>${dks_stdout}

# показываем
fn_file_echo ${dks_stdout} ${emptEsh}
fn_file_clr ${dks_stdout}

# Анализируем наличие FireWall и пытаемся добавить правило для открытия порта
# только для установки с http севрером
if ! [ "${ws_type}" = "no_http"  ]; then
  bash ${dks_bashfile_dir}/sys/sys_firewall.sh "${ws_port}" "tcp"
fi;


#systemctl stop ${dks_serv_name}
systemctl start ${dks_serv_name}

# === Настройка ротации логов
chown -R ks-www-data:ks-www-data ${inst_dir}/log
bash  ${dks_bashfile_dir}/sys/sys_logrotate.sh "${inst_dir}/log"        \
                                               "${tag_name}_${ws_port}" \
                                               "${wsks_msg_title}"      \
                                               "ks-www-data"

# === Проверяем что служба запустилась ============================================================

if systemctl -q is-active ${dks_serv_name}; then
  echo -e "${sOK}Запуск сервиса ${dks_serv_name}."

  if [ -z "${rebuild}" ] ; then
    systemctl stop ${dks_serv_name}
    echo -e "${wOK}Остановка сервиса ${dks_serv_name}."
    echo -e "${emptE}Отсутствуют установленные приложения."
    echo -e "${emptE}Для развертывания веб приложений используйте скрипт ws_addapp.sh."
    echo -e "${emptE}См. инструкцию по развертыванию ${inst_dir}/ctl/readme.txt"
  fi;
else
  echo -e "${eRR}Запуск сервиса ${dks_serv_name}."
  echo -e "${emptE}Была предпринята неудачная попытка запуска сервиса ${dks_serv_name}."
  echo -e "${emptE}Если сервис не будет запущен, то контейнер ${dks_cont_name} не запустится после перезагрузки хоста."
  echo -e "${emptE}Обязательно свяжитесь со своим системным администратором."
fi;

# восстанавливаем значение для selinux
if [ "${se_mode}" = "Enforcing" ]; then
  # переключаем обратно.
  setenforce Enforcing
fi;

rm ${dks_stdout}

echo ""
exit 0;
