#!/bin/bash

###
### Утилита работы с СУБД-КС
###
### Copyright (c) 2023, ООО Кейсистемс
###
### Версия 20230220
### Ссылка на актуальную версию https://keysystems.ru/files/web/Scripts/PG/pgsqlks_utils.sh

### Максимальное время ожидания выполнения запросов в СУБД в сек, 36000 сек = 10 часов
time_out=36000

# Цвета
ErrColor='\033[1;31m'
SuccColor='\033[0;32m'
WarnColor='\033[0;33m'
NoColor='\033[0m'

sOK="[===${SuccColor}OK${NoColor}===]="
wOK="[==${WarnColor}Инфо${NoColor}==]="
eRR="[=${ErrColor}Ошибка${NoColor}=]="
emptE="==========="


clear
echo ""
echo -e "${WarnColor}========================================================================================${NoColor}"
echo -e "${WarnColor}=                               Утилита работы с СУБД-КС                               =${NoColor}"
echo -e "${WarnColor}========================================================================================${NoColor}"
echo ""

if [[ $EUID != 0 ]]; then
    echo -e "${eRR}Скрипт должен выполняться root правами!"
    exit 1;
fi;

function wget_error() {
    if [ $? != "0" ]; then
       echo ""
       echo -e "${eRR}Ошибка скачивания $1"
       exit 1;
    fi;
}

function fn_file_clr(){
cat > $1 << EOF
EOF
}

function auth_get(){
 read -p "Введите пользователя с правами sysadm (dbo): " pg_usr
 read -p "            Введите пароль пользователя (1): " -s pg_pwd
 if [[ -z "${pg_usr}" ]]; then
   pg_usr='dbo'
 fi;
 if [[ -z "${pg_pwd}" ]]; then
   pg_pwd='1'
 fi;
 echo ""
 #echo " Пользователь: ${pg_usr} пароль: ${pg_pwd}"
}

####Выбор БД СУБД-КС
function pgsqlks_base() {
    bash_cmd=$(bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out}  -U ${pg_usr} -pwd ${pg_pwd} -c "SELECT datname FROM pg_database ORDER BY datname asc;" -k "-t"  | tr -cd '[:print:]' | sed 's/^[[:space:]]//g')
    if [ -z "$(echo ${bash_cmd} | grep "Docker_sql_command_completed")" ]; then
        echo ""
        echo -e "${eRR}Ошибка поиска БД в инстанс! Выполнение прервано!"
        exit 1;
    fi;
    bash_cmd=$(echo ${bash_cmd} | sed 's/\[.*//g' | tr ' ' '| ')
    IFS='|' read -r -a DbNames <<< ${bash_cmd} #нумерация массива с 0
    unset IFS
    DbNames_count=$((${#DbNames[*]}))
    DbNames[$(($DbNames_count+1))]="Выход"
    echo -e "${wOK}В выбранном инстанс СУБД-КС $pgsqlks_inst найдено БД всего: $DbNames_count"

    PS3="Выберите БД: "
    select menu in "${DbNames[@]}" ; do
       if ((${REPLY}<=${DbNames_count})) ; then
         DbName=$(echo $menu | tr -d '[:space:]')
         break
       else
          if ((${REPLY}==$(($DbNames_count+1)))) ; then
             exit
          else
             echo "Недопустимое действие $REPLY"
          fi;
       fi;
    done
}

####Выбор инстанса СУБД-КС
function pgsqlks_instance() {
    pgsqlks_servs="$(echo $(systemctl --type=service | grep 'pgsqlks-' | sed -e 's/^ *//' | cut -f1,2 -d"."))"
    IFS=' ' read -r -a pgsqlks_servs <<< "$pgsqlks_servs"
    unset IFS

    pgsqlks_count=${#pgsqlks_servs[*]}
    if [ $pgsqlks_count == 0 ]; then
        echo -e "${eRR}Инстансы СУБД-КС не найдены, возможно это не сервер СУБД-КС!"
        exit 1;
    fi;

    echo "Всего найдено $pgsqlks_count инстанса(ов) СУБД-КС:"
    pgsqlks_servs[$(($pgsqlks_count+1))]="Выход"

    PS3="Выберите сервер СУБД-КС: "
    select menu in "${pgsqlks_servs[@]}" ; do
       if ((${REPLY}<=${pgsqlks_count})) ; then
           pgsqlks_inst=$menu
           break
       else
           if ((${REPLY}==$(($pgsqlks_count+1)))) ; then
                exit
             else
                echo "Недопустимое действие $REPLY"
          fi;
       fi;
    done
    echo ""
    echo -e "${wOK}Выбран инстанс СУБД-КС: $pgsqlks_inst"
    echo ""
    pgsqlks_ver=$(echo $pgsqlks_inst | cut -f1 -d"_")
    pgsqlks_port=$(echo $pgsqlks_inst | cut -f2 -d"_")
    pgsqlks_data_path=$(bash /opt/dks-$pgsqlks_ver/sh/dks_init_var_$pgsqlks_port.sh 'db_data_path')
    pgsqlks_kslib_path=$(bash /opt/dks-$pgsqlks_ver/sh/dks_const_var.sh 'db_kslib_path')
    pgsqlks_tag_ver=$(bash /opt/dks-$pgsqlks_ver/sh/dks_const_var.sh 'tag_ver')
    pgsqlks_db_port=$(bash /opt/dks-$pgsqlks_ver/sh/dks_init_var_$pgsqlks_port.sh 'db_port')
    auth_get
    bash_cmd=$(bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "SELECT * from information_schema.table_privileges WHERE grantee = 'postgres' LIMIT 5;" -k "-t"  | tr -cd '[:print:]' | sed 's/^[[:space:]]//g')
    if [ -z "$(echo ${bash_cmd} | grep "Docker_sql_command_completed")" ]; then
        echo ""
        echo -e "${eRR}Ошибка соединения с $pgsqlks_inst! Выполнение прервано!"
        exit 1;
    fi;
}


### Backup БД СУБД-КС
function pgsqlks_backup() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                               Backup БД СУБД-КС                                      =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""

    pgsqlks_instance #Выбор инстанса СУБД-КС
    pgsqlks_base #Выбор БД СУБД-КС
    back_file="${DbName}_$(date "+%Y%m%d_%H%M%S").bak"

    echo ""
    echo -e "${wOK}Выбрана БД: $DbName"
    echo ""
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor} Скрипт получил следующие параметры backup         ${NoColor}"
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor}    Инстанс СУБД-КС: $pgsqlks_inst"
    echo -e "${WarnColor}             Имя БД: ${DbName}"
    echo -e "${WarnColor}Имя резервной копии: $pgsqlks_data_path/backups/$back_file"
    echo ""
    read -p "Вы уверены, что параметры верны? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1
    read -p "Продолжить backup БД? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1

    begin_time=$(date +'%d.%m.%Y %H:%M:%S')
    begin_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи начато: ${begin_time} ..."
    echo ""

    echo ""
    echo ""
    echo -e "${wOK}Создание backup начато..."
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "select xp_pg_dump ( v_DbName := '${DbName}', v_Path := '$pgsqlks_data_path/backups' ,v_File := '$back_file');" | grep -v "^[[:space:]]*$"

    end_time=$(date +'%d.%m.%Y %H:%M:%S')
    end_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи завершено: ${end_time} ..."
    echo -e "${wOK}    Потраченное время в мин: $(( $(( $end_sec-$begin_sec ))/60 ))"
    echo ""
}

### Восстановление БД из backup СУБД-КС
function pgsqlks_restore() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                         Восстановление БД из backup СУБД-КС                          =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""

    pgsqlks_instance #Выбор инстанса СУБД-КС

    IFS=';' read -r -a back_files <<< $(find ${pgsqlks_data_path}/backups -maxdepth 1 -name "*.bak" -printf "%f\n" | sort | tr '\n' ';')
    unset IFS
    back_files_count=${#back_files[*]}
    if [ $back_files_count == 0 ]; then
        echo -e "${eRR}Резервных копий БД (файлы с расширением 'bak') в $pgsqlks_data_path/backups не найдено!"
        exit 1;
    fi;

    echo "Найдено всего $back_files_count резервных копий в $pgsqlks_data_path/backups:"
    back_files[$(($back_files_count+1))]="Выход"

    PS3="Выберите резервную копию БД: "
    select menu in "${back_files[@]}" ; do
        if ((${REPLY}<=${back_files_count})) ; then
            back_file=$menu
            break
        else
            if ((${REPLY}==$(($back_files_count+1)))) ; then
               exit
            else
               echo "Недопустимое действие $REPLY"
            fi;
        fi;
    done

    echo ""
    echo -e "${wOK}Выбрана резервная копия: $back_file"
    echo ""

    echo -e "${wOK}В выбранном инстанс СУБД-КС $pgsqlks_inst найдены следующие БД:"
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "SELECT datname FROM pg_database ORDER BY datname asc;" | grep -v "^[[:space:]]*$"

    read -p "Укажите имя БД куда будем восстановливать резервную копию $back_file:" DbName
    DbName=$(echo $DbName | tr -d '[:space:]') #удаляем все пробелы
    if [ -z "${DbName}" ]; then
        echo ""
        echo -e "${eRR}Имя БД не указано! Выполнение прервано!"
        exit 1;
    fi;

    db_exist_str=$(bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "SELECT datname FROM pg_database WHERE datname='${DbName}';" | grep '0 rows')
    if [ -z "${db_exist_str}" ]; then
        echo ""
        echo -e "${eRR}БД с именем ${DbName} существует!"
        confirm_random=$(cat /dev/urandom | tr -dc A-Z0-9 | head -c 8; echo)
        read -p "Для продолжения восстановления введите следующий код $confirm_random : " confirm
        if [[ $confirm != $confirm_random ]]; then
           echo ""
           echo -e "${eRR}Введен некорректный код! Выполнение прервано!"
           exit 1;
        fi;
        db_exist=1
    else
        db_exist=0
    fi;

    echo ""
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor} Скрипт получил следующие параметры восстановления ${NoColor}"
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor}    Инстанс СУБД-КС: $pgsqlks_inst"
    echo -e "${WarnColor}Имя резервной копии: $pgsqlks_data_path/backups/$back_file"
    echo -e "${WarnColor}             Имя БД: ${DbName}"
    echo ""
    read -p "Вы уверены, что параметры верны? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1
    read -p "Продолжить восстановление БД? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1

    begin_time=$(date +'%d.%m.%Y %H:%M:%S')
    begin_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи начато: ${begin_time} ..."
    echo ""

    if [[ $db_exist == 1 ]]; then
        echo ""
        echo ""
        echo -e "${wOK}Удаление БД начато..."
        bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "DROP DATABASE IF EXISTS ${DbName} WITH (FORCE);" | grep -v "^[[:space:]]*$"
    fi;

    echo ""
    echo ""
    echo -e "${wOK}Создание БД начато..."
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "select xp_pg_createdb  (v_DbName := '${DbName}');" | grep -v "^[[:space:]]*$"

    echo ""
    echo ""
    echo -e "${wOK}Восстановление БД начато..."
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "select xp_pg_restore   (v_DbName := '${DbName}', v_Path := '$pgsqlks_data_path/backups/', v_File := '$back_file', v_Option := 1);" | grep -v "^[[:space:]]*$"

    #echo ""
    #echo ""
    #echo -e "${wOK}Начато \"vacuum analyze\" ..."
    #bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "vacuum analyze;" | grep -v "^[[:space:]]*$"

    end_time=$(date +'%d.%m.%Y %H:%M:%S')
    end_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи завершено: ${end_time} ..."
    echo -e "${wOK}    Потраченное время в мин: $(( $(( $end_sec-$begin_sec ))/60 ))"
    echo ""
}

### Удаление БД СУБД-КС
function pgsqlks_drop() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                               Удаление БД СУБД-КС                                    =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""

    pgsqlks_instance #Выбор инстанса СУБД-КС
    pgsqlks_base #Выбор БД СУБД-КС

    echo ""
    echo -e "${wOK}Выбрана БД: $DbName"
    echo ""
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor} Скрипт получил следующие параметры удаления       ${NoColor}"
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor}    Инстанс СУБД-КС: $pgsqlks_inst"
    echo -e "${WarnColor}             Имя БД: ${DbName}"
    echo ""
    read -p "Вы уверены, что параметры верны? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1
    read -p "Продолжить удаление БД? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1

    confirm_random=$(cat /dev/urandom | tr -dc A-Z0-9 | head -c 8; echo)
    read -p "Для удаления БД введите следующий код $confirm_random : " confirm
    if [[ $confirm != $confirm_random ]]; then
       echo ""
       echo -e "${eRR}Введен некорректный код! Выполнение прервано!"
       exit 1;
    fi;

    begin_time=$(date +'%d.%m.%Y %H:%M:%S')
    begin_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи начато: ${begin_time} ..."
    echo ""

    echo ""
    echo ""
    echo -e "${wOK}Удаление БД $DbName начато..."
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "DROP DATABASE IF EXISTS $DbName WITH (FORCE);" | grep -v "^[[:space:]]*$"
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "DROP ROLE IF EXISTS "ks_connect_${DbName}";" | grep -v "^[[:space:]]*$"

    end_time=$(date +'%d.%m.%Y %H:%M:%S')
    end_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи завершено: ${end_time} ..."
    echo -e "${wOK}    Потраченное время в мин: $(( $(( $end_sec-$begin_sec ))/60 ))"
    echo ""
}

### Выполнить скрипт из файла
function pgsqlks_script_exec() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                               Выполнить скрипт из файла                              =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""

    pgsqlks_instance #Выбор инстанса СУБД-КС
    pgsqlks_base #Выбор БД СУБД-КС

    echo ""
    echo -e "${wOK}Выбрана БД: $DbName"
    echo ""

    IFS=';' read -r -a sql_files <<< $(find ${dks_bashfile_dir} -maxdepth 1 -name "*.sql" -printf "%f\n" | sort | tr '\n' ';')
    unset IFS
    sql_files_count=${#sql_files[*]}
    if [ $sql_files_count == 0 ]; then
        echo -e "${eRR}Скрипты (файлы с расширением 'sql') в $dks_bashfile_dir не найдено!"
        exit 1;
    fi;

    echo "Найдено всего $sql_files_count скрипт(а/ов) по пути $dks_bashfile_dir:"
    sql_files[$(($sql_files_count+1))]="Выход"

    PS3="Выберите скрипт *.sql: "
    select menu in "${sql_files[@]}" ; do
        if ((${REPLY}<=${sql_files_count})) ; then
            sql_file=$menu
            break
        else
            if ((${REPLY}==$(($sql_files_count+1)))) ; then
               exit
            else
               echo "Недопустимое действие $REPLY"
            fi;
        fi;
    done

    echo ""
    echo -e "${wOK}Содержимое скрипта $dks_bashfile_dir/$sql_file :"
    echo    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    sed  -n '1,3p' $dks_bashfile_dir/$sql_file && echo
    echo -e "...............                                   "
    echo -e "...............                                   "
    echo -e "...............                                   "
    tail -3 $dks_bashfile_dir/$sql_file
    echo -e "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    echo ""
    echo -e "${WarnColor}------------------------------------------------------------${NoColor}"
    echo -e "${WarnColor} Требуется выполнить SQL-скрипт со следующими параметрами   ${NoColor}"
    echo -e "${WarnColor}------------------------------------------------------------${NoColor}"
    echo -e "${WarnColor}    Инстанс СУБД-КС: $pgsqlks_inst"
    echo -e "${WarnColor}             Имя БД: ${DbName}"
    echo -e "${WarnColor}          SQL-срипт: $dks_bashfile_dir/$sql_file"
    echo ""
    read -p "Вы уверены, что параметры верны? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1

    begin_time=$(date +'%d.%m.%Y %H:%M:%S')
    begin_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи начато: ${begin_time} ..."
    echo ""
    echo -e "${wOK}Выполнение скрипта начато..."
    cp $dks_bashfile_dir/$sql_file ${pgsqlks_data_path}
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d ${DbName} -f "${pgsqlks_data_path}/$sql_file" | grep -v "^[[:space:]]*$" > $dks_bashfile_dir/$sql_file.log
    rm -f ${pgsqlks_data_path}/$sql_file

    end_time=$(date +'%d.%m.%Y %H:%M:%S')
    end_sec=$(date +%s)

    if [[ ! -z "$(grep -i "Docker_sql_command_error" $dks_bashfile_dir/$sql_file.log)" ]]; then
       echo -e "${eRR}Ошибка выполнения скрипта. Лог-файл: $dks_bashfile_dir/$sql_file.log"
    else
       echo -e "${sOK}Скрипт выполнен успешно. Лог-файл: $dks_bashfile_dir/$sql_file.log"
    fi;
    echo ""
    echo -e "${wOK}Выполнение задачи завершено: ${end_time} ..."
    echo -e "${wOK}    Потраченное время в мин: $(( $(( $end_sec-$begin_sec ))/60 ))"
    echo ""
}


### Переименовать БД
function pgsqlks_db_rename() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                               Переименовать БД                                       =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""

    pgsqlks_instance #Выбор инстанса СУБД-КС
    pgsqlks_base #Выбор БД СУБД-КС

    echo ""
    echo -e "${wOK}Выбрана БД: $DbName"
    echo ""
    read -p "Укажите новое имя БД:" DbName_new
    DbName_new=$(echo $DbName_new | tr -d '[:space:]') #удаляем все пробелы
    if [ -z "${DbName_new}" ]; then
       echo ""
       echo -e "${eRR}Имя БД не указано! Выполнение прервано!"
       exit 1;
    fi;

    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor} Скрипт получил следующие параметры переименования ${NoColor}"
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor}    Инстанс СУБД-КС: $pgsqlks_inst"
    echo -e "${WarnColor}     Текущее имя БД: ${DbName}"
    echo -e "${WarnColor}       Новое имя БД: ${DbName_new}"
    echo ""
    read -p "Вы уверены, что параметры верны? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1
    read -p "Продолжить переименование БД? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1

    cmd_sessions="select pid as session_id
    , left(datname::varchar,30) as database_name
    , left(usename,30) as login_name
    , client_addr::varchar as host_name
    , application_name::varchar as app_name
    , state as status
    , cast(query_start as varchar) as start_time
    , backend_xid::varchar as open_tran
    , 0::int as cpu
    from pg_stat_activity where datid is not null and datname='${DbName}'
    order by status;"

    begin_time=$(date +'%d.%m.%Y %H:%M:%S')
    begin_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи начато: ${begin_time} ..."
    echo ""

    db_sessions=$(bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "${cmd_sessions}")
    db_sessions_res=$(echo ${db_sessions} | grep '0 rows')
    if [ -z "${db_sessions_res}" ]; then
       echo -e "${eRR}Найдены открытые сессии, переименование отменено!"
       echo ""
       bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "${cmd_sessions}" | grep -v "^[[:space:]]*$"
       exit 1;
    fi;

    cmd_rename="do
     \\\$\$
     begin
        alter database "${DbName}" rename to "${DbName_new}";
        drop role if exists "ks_connect_${DbName_new}";
        if exists(select 1 from pg_roles where rolname='ks_connect_${DbName}') then
           --ks_connect_<DATABASE> существует
           alter role "ks_connect_${DbName}" rename to "ks_connect_${DbName_new}";
        else
           --создаем ks_connect_<DATABASE>
           create role "ks_connect_${DbName_new}" nosuperuser nocreatedb nocreaterole nologin noreplication;
        end if;
     end;
     \\\$\$;"

    echo ""
    echo ""
    echo -e "${wOK}Переименование БД $DbName начато..."
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "${cmd_rename}" | grep -v "^[[:space:]]*$" > "$dks_bashfile_dir/pgsqlks_dbrename.log"
    if [[ ! -z "$(grep -i "Docker_sql_command_error" $dks_bashfile_dir/pgsqlks_dbrename.log)" ]]; then
       echo -e "${eRR}Ошибка переименования БД."
       cat "$dks_bashfile_dir/pgsqlks_dbrename.log"
       echo -e "Открытые сессии:"
       bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "${cmd_sessions}" | grep -v "^[[:space:]]*$"
    else
       echo -e "${sOK}БД переименована успешно."
    fi;
    rm -f $dks_bashfile_dir/pgsqlks_dbrename.log

    end_time=$(date +'%d.%m.%Y %H:%M:%S')
    end_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи завершено: ${end_time} ..."
    echo -e "${wOK}    Потраченное время в мин: $(( $(( $end_sec-$begin_sec ))/60 ))"
    echo ""
}

####Информация о СУБД-КС
function pgsqlks_info() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                              Информация о СУБД-КС                                    =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""
    pgsqlks_instance #Выбор инстанса СУБД-КС

    echo -e ""
    echo -e "${wOK}Расположение: $pgsqlks_data_path"
    IFS=' ' read -r -a data_info <<< "$(df -h $pgsqlks_data_path | head -n2 | tail -n1)"
    unset IFS
    data_info_count=${#data_info[*]}
    if [ $data_info_count != 0 ]; then
       echo "             Устройство: ${data_info[0]}"
       echo "                 Размер: ${data_info[1]}"
       echo "           Использовано: ${data_info[2]}"
       echo "               Доступно: ${data_info[3]}"
       echo "        Использовано(%): ${data_info[4]}"
       echo "          Монтировано в: ${data_info[5]}"
    fi;

    echo ""
    echo -e "${wOK}Версия СУБД-КС: $(bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -k "-t " -c 'SHOW server_version;' | grep -v 'Docker_sql_command_completed' | sed 's/Ожидание завершения выполнения SQL команды.*сек.)//' | tr -d '[:space:]')"
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -k "-t " -c "SELECT version();" | grep -v "^[[:space:]]*$" | grep -v "Docker_sql_command_completed" | sed 's/Ожидание завершения выполнения SQL команды.*сек.)//'

    echo ""
    echo -e "${wOK}Расположение kslib: ${pgsqlks_kslib_path}/${pgsqlks_db_port}"

    echo ""
    tmp="$(bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -k "-t " -c 'SELECT pg_size_pretty(sum(pg_database_size(datname))) from pg_database;' | grep -v "^[[:space:]]*$" | grep -v "Docker_sql_command_completed" | sed 's/Ожидание завершения выполнения SQL команды.*сек.)//')"
    echo -e "${wOK}Базы данных: (размер всего ${tmp})"
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "SELECT datname As Имя, pg_size_pretty(pg_database_size(pg_database.datname)) As Размер, datacl As Права_доступа FROM pg_database ORDER BY datname asc;" | grep -v "^[[:space:]]*$" | grep -v "Docker_sql_command_completed" | sed 's/Ожидание завершения выполнения SQL команды.*сек.)//'

    echo ""
    echo -e "${wOK}Роли и пользователи выгружены в $dks_bashfile_dir/roles.txt"
    echo -e " Инстанс СУБД-КС: $pgsqlks_inst"  > $dks_bashfile_dir/roles.txt
    echo -e " Роли и пользователи:"  >> $dks_bashfile_dir/roles.txt
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "SELECT * FROM pg_roles ORDER BY rolname;" >> $dks_bashfile_dir/roles.txt
}


#### Обновление библиотеки libxpks
function pgsqlks_xpks() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                          Обновление библиотек xpks                                   =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""
    pgsqlks_instance #Выбор инстанса СУБД-КС

    date_tmp=$(date "+%Y%m%d")
    pgsqlks_tag_ver1=$(echo ${pgsqlks_tag_ver} | awk -F '.' '{print $1}') #определяем версию PG разделенную точкой 14.6
    echo -e "${wOK}Версия СУБД-КС: ${pgsqlks_tag_ver} (${pgsqlks_tag_ver1})"

    mkdir -p ${dks_bashfile_dir}/xpks_${date_tmp}
    echo ""
    echo -e "${wOK}Начато скачивание библиотеки..."

    wget -O ${dks_bashfile_dir}/xpks_${date_tmp}/libxpks.so "https://update.keysystems.ru/SrvUpdateService/Update.mvc/GetXPKSFile?path=xpks%2FLINUX%2FX64%2F${pgsqlks_tag_ver1}%2Flibxpks.so&db=2" --no-check-certificate --no-verbose || wget_error libxpks.so
    wget -O ${dks_bashfile_dir}/xpks_${date_tmp}/pg_dump "https://update.keysystems.ru/SrvUpdateService/Update.mvc/GetXPKSFile?path=xpks%2FLINUX%2Fpg_dump&db=2" --no-check-certificate --no-verbose || wget_error pg_dump
    wget -O ${dks_bashfile_dir}/xpks_${date_tmp}/pg_restore "https://update.keysystems.ru/SrvUpdateService/Update.mvc/GetXPKSFile?path=xpks%2FLINUX%2Fpg_restore&db=2" --no-check-certificate --no-verbose || wget_error pg_restore
    tar -cvzf ${dks_bashfile_dir}/xpks_${date_tmp}/kslib_$(date "+%Y%m%d_%H%M%S").tar.gz --directory ${pgsqlks_kslib_path} ${pgsqlks_db_port}
    echo -e "${wOK}Создан архив старой библиотеки и скачаны новые версии libxpks в: ${dks_bashfile_dir}/xpks_${date_tmp}"

    owner_group=$(stat -c %G ${pgsqlks_kslib_path}/${pgsqlks_db_port}) #определяем владельца каталога kslib
    owner_user=$(stat -c %U ${pgsqlks_kslib_path}/${pgsqlks_db_port})

    mv -f ${dks_bashfile_dir}/xpks_${date_tmp}/libxpks.so ${pgsqlks_kslib_path}/${pgsqlks_db_port}/libxpks.so
    chown ${owner_user}:${owner_group} ${pgsqlks_kslib_path}/${pgsqlks_db_port}/libxpks.so
    chmod 775 ${pgsqlks_kslib_path}/${pgsqlks_db_port}/libxpks.so

    mv -f ${dks_bashfile_dir}/xpks_${date_tmp}/pg_dump ${pgsqlks_kslib_path}/${pgsqlks_db_port}/pg_dump
    chown ${owner_user}:${owner_group} ${pgsqlks_kslib_path}/${pgsqlks_db_port}/pg_dump
    chmod 775 ${pgsqlks_kslib_path}/${pgsqlks_db_port}/pg_dump

    mv -f ${dks_bashfile_dir}/xpks_${date_tmp}/pg_restore ${pgsqlks_kslib_path}/${pgsqlks_db_port}/pg_restore
    chown ${owner_user}:${owner_group} ${pgsqlks_kslib_path}/${pgsqlks_db_port}/pg_restore
    chmod 775 ${pgsqlks_kslib_path}/${pgsqlks_db_port}/pg_restore
    echo -e "${wOK}Библиотека libxpks обновлена в: ${pgsqlks_kslib_path}/${pgsqlks_db_port}"
}



### Копия БД на основе выбранного
function pgsqlks_copydb() {
    echo ""
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo -e "${WarnColor}=                              Копия БД на основе выбранного                           =${NoColor}"
    echo -e "${WarnColor}========================================================================================${NoColor}"
    echo ""

    pgsqlks_instance #Выбор инстанса СУБД-КС
    pgsqlks_base #Выбор БД СУБД-КС
    DbName_new="${DbName}_test"
    echo ""
    echo -e "${wOK}Выбрана БД: $DbName"
    echo ""

    read -p "Укажите имя копии БД (по умолчанию ${DbName_new}):" DbName_new_tmp
    DbName_new_tmp=$(echo $DbName_new_tmp | tr -d '[:space:]') #удаляем все пробелы
    if [[ ! -z "${DbName_new_tmp}" ]]; then
       DbName_new=$DbName_new_tmp
    fi;

    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor} Скрипт получил следующие параметры создания копии ${NoColor}"
    echo -e "${WarnColor}---------------------------------------------------${NoColor}"
    echo -e "${WarnColor}    Инстанс СУБД-КС: $pgsqlks_inst"
    echo -e "${WarnColor}             Имя БД: ${DbName}"
    echo -e "${WarnColor}       Имя копии БД: ${DbName_new}"
    echo ""
    read -p "Вы уверены, что параметры верны? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1
    read -p "Продолжить создание копии БД? (Y/N): " confirm && [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]] || exit 1

    begin_time=$(date +'%d.%m.%Y %H:%M:%S')
    begin_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи начато: ${begin_time} ..."
    echo ""
    echo -e "${wOK}Создание копии БД начато..."
    bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -c "select xp_pg_copydb(v_DbName := '${DbName}', v_DbNameCopy := '${DbName_new}', v_PathBackup :='', v_Option := '1'::integer);" | grep -v "^[[:space:]]*$"

    end_time=$(date +'%d.%m.%Y %H:%M:%S')
    end_sec=$(date +%s)
    echo ""
    echo -e "${wOK}Выполнение задачи завершено: ${end_time} ..."
    echo -e "${wOK}    Потраченное время в мин: $(( $(( $end_sec-$begin_sec ))/60 ))"
    echo ""
}


### Обслуживание БД
function pgsqlks_db_service() {
    pgsqlks_instance #Выбор инстанса СУБД-КС
    echo ""
    pgsqlks_base #Выбор БД СУБД-КС

    echo ""
 while true; do
    local MENU=(
        "Дефрагментация"
        "Обновление статистики"
        "Переиндексация БД (мон)"
        "Сжатие (мон)"
        "Назад"
    )
    local PS3="Выберите действия в СУБД-КС $pgsqlks_inst в БД $DbName: "
    echo -e "${wOK}Обслуживание БД (мон-действия требуют монопольный доступ)"
    select menu in "${MENU[@]}" ; do
       case $REPLY in
        1) echo ""
           echo -e "${wOK}Начато \"$menu\" ..."  | tee -a ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "SELECT session_user, current_database();" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "Vacuum verbose;" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           echo -e "${sOK}Лог-файл \"$menu\": ${dks_bashfile_dir}/${bashfile_log_name}"
           echo ""
           break
           ;;
        2) echo ""
           echo -e "${wOK}Начато \"$menu\" ..."  | tee -a ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "SELECT session_user, current_database();" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "analyze verbose;" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           echo -e "${sOK}Лог-файл \"$menu\": ${dks_bashfile_dir}/${bashfile_log_name}"
           echo ""
           break
           ;;
        3) echo ""
           echo -e "${wOK}Начато \"$menu\" ..."  | tee -a ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "SELECT session_user, current_database();" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "reindex database $DbName;" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           echo -e "${sOK}Лог-файл \"$menu\": ${dks_bashfile_dir}/${bashfile_log_name}"
           echo ""
           break
           ;;
        4) echo ""
           echo -e "${wOK}Начато \"$menu\" ..."  | tee -a ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "SELECT session_user, current_database();" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           bash /opt/$pgsqlks_ver/utility/$pgsqlks_port/psql.sh -h 127.0.0.1 -t ${time_out} -U ${pg_usr} -pwd ${pg_pwd} -d $DbName -c "Vacuum full verbose;" | grep -v "^[[:space:]]*$" >> ${dks_bashfile_dir}/${bashfile_log_name}
           echo -e "${sOK}Лог-файл \"$menu\": ${dks_bashfile_dir}/${bashfile_log_name}"
           echo ""
           break
           ;;
        5) return
           ;;
        *) echo "Недопустимое действие $REPLY" ;;
       esac
    done
 done
}


### BEGIN
bashfile_name="${0##*/}"
bashfile_log_name="${0##*/}.log"
dks_bashfile_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

echo "Скрипт \"${bashfile_name}\" запущен: $(date +'%d.%m.%Y %H:%M:%S')" > ${dks_bashfile_dir}/${bashfile_log_name}

while true; do
 MENU=(
    "Создать backup БД"
    "Восстановить БД из backup"
    "Выполнить скрипт из файла"
    "Информация о СУБД-КС"
    "Удалить БД"
    "Переименовать БД"
    "Создать копию БД на основе выбранного"
    "Обслуживание БД"
    "Обновить библиотеку libxpks"
    "Выход"
 )

 PS3="Выберите действие: "
 echo ""
 echo -e "${wOK}ОСНОВНОЕ МЕНЮ"
 select menu in "${MENU[@]}" ; do
   case $REPLY in
       1) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_backup
             exit;
          fi;
          ;;
       2) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_restore
             exit;
          fi;
          ;;
       3) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_script_exec
             exit;
          fi;
          ;;
       4) action=$REPLY
          pgsqlks_info
          break;
          ;;
       5) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_drop
             exit;
          fi;
          ;;
       6) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_db_rename
             exit;
          fi;
          ;;
       7) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_copydb
             exit;
          fi;
          ;;
       8) action=$REPLY
          echo ""
          pgsqlks_db_service
          break
          ;;
       9) action=$REPLY
          read -p "Вы уверены? (Y/N): " confirm
          if [[ $confirm == [yY] || $confirm == [yY][eE][sS] ]]; then
             pgsqlks_xpks
          fi;
          break;
          ;;
       10) exit
          ;;
       *) echo "Недопустимое действие $REPLY" ;;
   esac
 done
done
