#!/bin/bash

### СУБД-КС "K"
###
### Copyright (c) 2024, ООО Кейсистемс
###
### Версия 14.13

# Определяем каталог выполнения текущего скрипта
dks_bashfile_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

# Параметры:
#                     $1 - порт (обязательный)
#                     $2 - пароль для postgres и dbo    (не обязательный, если пусто, то=1)
#                     $3 - размер shared memory при создании контейнера, по умолчанию в
#                          в самом докере 64MB, мы же ставим свое значение 256MB
#                     $4 - Локаль для инициализации сервера СУБД
#                     $5 - ПО контейнеризации



typeOS="ksdeb11"

if [ -z "$3" ]; then
  shm_size="256"
else
  shm_size=$3
fi;

locale=$4
contsoft=$5


#============== Основные переменные константы =====================
tag_ver=$($dks_bashfile_dir/dks_const_var.sh tag_ver)
tag_name=$($dks_bashfile_dir/dks_const_var.sh tag_name)


# Контейнер
dks_inst_name=$($dks_bashfile_dir/dks_const_var.sh dks_inst_name)
dks_inst_path=$($dks_bashfile_dir/dks_const_var.sh dks_inst_path)

# СУБД
db_inst_name=$($dks_bashfile_dir/dks_const_var.sh db_inst_name)
db_bin_path=$($dks_bashfile_dir/dks_const_var.sh db_bin_path)
db_lib_path=$($dks_bashfile_dir/dks_const_var.sh db_lib_path)
db_kslib_path=$($dks_bashfile_dir/dks_const_var.sh db_kslib_path)

# Переменные константы
net_access=$($dks_bashfile_dir/dks_const_var.sh net_access)
net_access_search=$($dks_bashfile_dir/dks_const_var.sh net_access_search)
net_dbo_access=$($dks_bashfile_dir/dks_const_var.sh net_dbo_access)

# Цвета
sOK=$(echo $($dks_bashfile_dir/dks_const_var.sh sOK) | tr '#' ' ')
wOK=$(echo $($dks_bashfile_dir/dks_const_var.sh wOK) | tr '#' ' ')
eRR=$(echo $($dks_bashfile_dir/dks_const_var.sh eRR) | tr '#' ' ')
emptE=$(echo $($dks_bashfile_dir/dks_const_var.sh emptE) | tr '#' ' ')
emptEsh=$($dks_bashfile_dir/dks_const_var.sh emptE)

# sudo
is_root=$($dks_bashfile_dir/dks_const_var.sh is_root)

function fn_file_clr(){
cat > $1 << EOF
EOF
}

function fn_file_echo() {
local LeftSpace=$2
LeftSpace=$(echo ${LeftSpace} | tr '#' ' ')
cat $1 | while read line
do
 echo -e "${LeftSpace}$line"
 done
}

# =========== Проверка параметров ===================================

echo ""
pgsqlks_msg_title="сервер СУБД"

# Проверка прав суперпользователя
bash ${dks_bashfile_dir}/sys/sys_is_root.sh "install"
if [ $? = "1" ]; then exit 1; fi;


if [ -z "$1" ]; then
  echo -e "${eRR}Проверка корректности значения указанного порта для сервера СУБД."
  echo -e "${emptE}Не указан порт для работы СУБД."
  echo -e "${emptE}Значение порта должно быть в диапазоне [5432-5444]."
  echo ""
  exit 1;
else
  # Проверка диапазона портов
  bash ${dks_bashfile_dir}/sys/sys_checkport.sh "interval" "$1" "${pgsqlks_msg_title}" "5432" "5444"
  if [ $? = "1" ]; then exit 1; fi;

  # Проверка "свободности" порта на хосте
  bash ${dks_bashfile_dir}/sys/sys_checkport.sh "free" "$1" "${pgsqlks_msg_title}"
  if [ $? = "1" ]; then exit 1; fi;

  # По каждому порту свой файл инициализации, который генерится на основе шаблона
  # скриптом dks_make_init_file.sh на базе 2 параметров порта и пароля сисадмина
  bash ${dks_bashfile_dir}/dks_make_init_file.sh "$1" "${contsoft}" "$2" "${typeOS}"
  dks_init_var_file="dks_init_var_$1.sh"
fi;


# ========== Переменные, которые зависят от порта  ====================


# Проверяем, что файл создался
if [ ! -f "${dks_bashfile_dir}/${dks_init_var_file}" ]; then
  echo -e "${eRR}Отсутствует файл инициализации переменных ${dks_bashfile_dir}/${dks_init_var_file}."
  echo -e "${emptE}Операция установки будет прервана."
  echo -e ""
  exit 1;
fi

# Проверка параметра shared memory (shm_size) для запуска контейнера
if ! ((64<=${shm_size} && ${shm_size}<=1024)); then
  echo -e "${eRR}Проверка корректности значения параметра контейнера shared memory."
  echo -e "${emptE}Некорректное значение \"${shm_size}\"."
  echo -e "${emptE}Значение параметра shared memory должно быть в диапазоне [64-1024]."
  echo -e "${emptE}Операция установки будет прервана."
  echo -e ""
  exit 1;
fi;
echo -e "${sOK}Проверка корректности значения \"${shm_size}\" параметра контейнера \"Shared Memory\"."

#  Проверка поддерживаемых локалей
isLCOK="no"
aLC_support=("be_BY.UTF-8" "en_US.UTF-8" "es_CU.UTF-8" "ru_RU.UTF-8" "tg_TJ.UTF-8" "be_BY.utf8" "en_US.utf8" "es_CU.utf8" "ru_RU.utf8" "tg_TJ.utf8")

# Проверка докали из массива доступных
case "${aLC_support[@]}" in  *"${locale}"*) isLCOK="yes" ;; esac

if [ "${isLCOK}" = "no" ] ; then
  echo -e "${eRR}Указана неподдерживаемая локаль \"${locale}\" для кластера СУБД."
  echo -e "${emptE}Значение локали должно быть: ${aLC_support[@]}."
  echo -e "${emptE}Операция установки будет прервана."
  echo -e ""
  exit 1;
fi;

# Проверка локали на хосте #################################
file_os_exists=$(which locale 2>/dev/null)
if [ -z ${file_os_exists} ]; then
  echo -e "${eRR}В системе отсутствует компонент \"locale\"."
  echo -e "${emptE}Операция установки будет прервана."
  exit 1;
fi;

grep_locales="";
for aLC_item in ${aLC_support[*]}
do
    if [ -z ${grep_locales} ]; then
        grep_locales=${aLC_item}
    else
        grep_locales=${grep_locales}'\|'${aLC_item}
    fi;
done;

exists_locale=$(locale -a | grep ${locale})
if [ -z "${exists_locale}" ]; then
  exists_locale=$(locale -a | grep "${grep_locales}")
  echo -e "${eRR}Указанная локаль \"${locale}\" не поддерживается на хосте."
  echo -e "${emptE}Доступные локали хоста: \"${exists_locale}\"."
  echo -e "${emptE}Операция установки будет прервана."
  echo -e ""
  exit 1;
fi;


# Контейнер
dks_cont_name=$($dks_bashfile_dir/${dks_init_var_file} dks_cont_name)
dks_serv_name=$($dks_bashfile_dir/${dks_init_var_file} dks_serv_name)
dks_image_tar_file=$(${dks_bashfile_dir}/${dks_init_var_file} dks_image_tar_file)

# СУБД
db_data_path=$($dks_bashfile_dir/${dks_init_var_file} db_data_path)
db_port=$($dks_bashfile_dir/${dks_init_var_file} db_port)
db_pwd=$(echo $($dks_bashfile_dir/${dks_init_var_file} db_pwd) | sed 's/"/\\"/g')

# Путь на хосте к kslib, делаем по порту, чтобы разнести xpks,
# т.к. при ее обновлении на одном инстансе приводит к крашу СУБД
# на других инстансах.
# Пусть для каждого иснтанса СУБД будет своя xpks в своем каталоге.
db_kslib_path_host="${db_kslib_path}/${db_port}"

util_path=$($dks_bashfile_dir/${dks_init_var_file} util_path)
repmgr_path=$($dks_bashfile_dir/${dks_init_var_file} repmgr_path)

# Файл обмена относительно хоста и контейнера
dks_stdout=$($dks_bashfile_dir/${dks_init_var_file} dks_stdout)
dks_stdout_cont=$($dks_bashfile_dir/${dks_init_var_file} dks_stdout_cont)
#==========================================================


# Проверка сервиса docker или файла podman на хосте
if [ "${contsoft}" = "docker"  ]; then
  contsoftname="Docker"
  consoftchecktype="service"
else
  contsoftname="Podman"
  consoftchecktype="file"
fi;

bash ${dks_bashfile_dir}/sys/sys_checkservice.sh "${contsoft}" "${contsoftname}" "${consoftchecktype}"
if [ $? = "1" ]; then exit 1; fi;

# Проверка , нет ли контейнера с указанным портом и другим именем
if [ ! -z "$(${contsoft} ps -a | grep ">${db_port}/tcp" | grep -v ${dks_cont_name})" ]; then
  echo -e "${eRR}Проверка доступности порта."
  echo -e "${emptE}Порт \"${db_port}\" используется другим контейнером."
  echo -e "${emptE}Операция установки будет прервана."
  echo ""
  exit 1;
fi

#= Проверка установки необходимых пакетов=================
# dmidecode
if ! [ -f /usr/sbin/dmidecode ]; then
  echo -e "${eRR}Не найден файл dmidecode."
  echo -e "${emptE}Выполните установку пакета dmidecode."
  echo -e "${emptE}Операция установки будет прервана."
  echo ""
  exit 1;
fi;

# localtime
if ! [ -f /etc/localtime ]; then
  echo -e "${eRR}Не найден файл localtime."
  echo -e "${emptE}Выполните установку пакета tzdata."
  echo -e "${emptE}Операция установки будет прервана."
  echo ""
  exit 1;
fi;


#= Подготовка к развертыванию окружения хоста==============
se_mode="Disabled"
# Проверка Selinux
if [ -f /etc/selinux/config ]; then
  se_mode=$(echo $(getenforce))
  if [ "${se_mode}" = "Enforcing" ]; then
    # переключаем в разрешительный :)
    setenforce Permissive
  fi;
fi;

# Удаление принудительно контейнер+образ #############################
${contsoft} container inspect ${dks_cont_name} >/dev/null 2>&1
if [ $? -eq 0 ]; then
  echo -e "${sOK}Удаление контейнера ${dks_cont_name}."
  ${contsoft} rm -f ${dks_cont_name} >/dev/null
fi;

${contsoft} image inspect ${dks_cont_name} >/dev/null 2>&1
if [ $? -eq 0 ]; then
  echo -e "${sOK}Удаление образа ${dks_cont_name}."
  ${contsoft} rmi -f ${dks_cont_name} >/dev/null
fi;

${contsoft} image inspect ${tag_name} >/dev/null 2>&1
if [ $? -eq 0 ]; then
  echo -e "${sOK}Удаление эталонного локального образа ${tag_name}."
  ${contsoft} rmi -f ${tag_name} >/dev/null
fi;
####################################################################


# Создание каталогов для монтирования для контейнера-докера
mkdir -p ${db_kslib_path}
mkdir -p ${db_kslib_path_host}
mkdir -p ${db_data_path}
mkdir -p ${repmgr_path}/${db_port}


# Создание утилит
bash ${dks_bashfile_dir}/dks_make_util.sh "${dks_init_var_file}"

# Создаем Файл обменана хосте
fn_file_clr ${dks_stdout}
chmod 777 ${dks_stdout}


# Копируем библиотеку xpks и всп. файлы
cp ${dks_inst_path}/kslib/* ${db_kslib_path_host}
chmod 777 ${db_kslib_path_host}/*

# Создание пользователя ks-postgres
fn_file_echo ${dks_stdout} ${emptEsh}

bash ${dks_bashfile_dir}/sys/sys_useradd.sh "ks-postgres"
if [ $? = "1" ]; then exit 1; fi;

#useradd postgres --user-group --create-home >/dev/null 2>${dks_stdout}
##показываем
#fn_file_echo ${dks_stdout} ${emptEsh}
#fn_file_clr ${dks_stdout}


# Владельцем делаем ks-postgres
chown ks-postgres:ks-postgres -R ${db_kslib_path}
chown ks-postgres:ks-postgres -R ${db_kslib_path_host}
chown ks-postgres:ks-postgres -R ${db_data_path}


# Загружаем образ из архива
if [ ! -f "${dks_inst_path}/container_image/${dks_image_tar_file}" ]; then
  echo -e "${eRR}Загрузка эталонного локального образа ${tag_name} из архива ${dks_image_tar_file}"
  echo -e "${emptE}Отсутствует файл архива с образом ${dks_inst_path}/container_image/${dks_image_tar_file}"
  echo -e "${emptE}Операция создания контейнера ${dks_cont_name} невозможна."
  echo -e "${emptE}Операция установки будет прервана."
  exit 1;
else
  ${contsoft} load -q -i "${dks_inst_path}/container_image/${dks_image_tar_file}" >/dev/null
  echo -e "${sOK}Загрузка эталонного локального образа ${tag_name} из архива ${dks_image_tar_file}."
fi


# Создаем на базе загруженного образа
${contsoft} build -q -t ${dks_cont_name} ${dks_inst_path}/container_image >/dev/null
echo -e "${sOK}Построение конечного образа ${dks_cont_name} на базе эталонного."


# Создаем контейнер из образа и запускаем в фоновом режиме
${contsoft} run -d \
           --privileged --device=/dev/mem  \
           --network host \
           --shm-size=${shm_size}MB \
           -v ${db_kslib_path_host}:${db_kslib_path}:rw,z \
           -v ${db_data_path}:${db_data_path}:rw,Z \
           -v ${repmgr_path}/${db_port}:/etc/repmgr:rw,Z \
           -v /etc/localtime:/etc/localtime:ro,z \
           -i \
           --name ${dks_cont_name} ${dks_cont_name} /bin/bash >/dev/null

echo -e "${sOK}Создание контейнера ${dks_cont_name} и запуск в ${contsoft}."


## -- инфа о запуске контейнера
# Определяем запущен ли контейнера
dks_status=$(${contsoft} inspect -f '{{.State.Running}}' ${dks_cont_name})

if [ "${dks_status}" = "true" ]; then
  echo -e "${sOK}Запуск контейнера ${dks_cont_name}."
fi;

if [ ! -z "$(${contsoft} images | grep ${tag_name})" ]; then
  echo -e "${sOK}Удаление эталонного локального образа ${tag_name}."
  ${contsoft} rmi -f ${tag_name} >/dev/null
fi;



# dmidecode находится в /usr/sbin
# для выполнения корректируем права и устанавливаем выполнение от имени root
# chown root:root /usr/sbin/dmidecode
# chmod u=rwx,g=rx,o=rx /usr/sbin/dmidecode
# chmod u+s /usr/sbin/dmidecode

bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh  -cs "${contsoft}"      \
                                                  -cn "${dks_cont_name}" \
                                                  -c  "chown root:root /usr/sbin/dmidecode && chmod u=rwx,g=rx,o=rx /usr/sbin/dmidecode && chmod u+s /usr/sbin/dmidecode" \
                                                  -noout


if [ $? = "1" ]; then exit 1; fi;

# Замена ID пользователя postgres на хостовый ID (считываем в переменные)
# Это нужно для того чтобы синхронизировать права доступа к каталогу PGDATA
# Пользователи с равными ID будут наследовать и права доступа
# Поэтому выравниваем id для пользователя postgres
# символ "/" экранируем в hex-16 -> \x2F

uid_host=$(id -u ks-postgres)
gid_host=$(id -g ks-postgres)

run_cmd="groupmod -g ${gid_host} postgres && usermod -u ${uid_host} -g ${gid_host} postgres"
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${run_cmd}" -noout
if [ $? = "1" ]; then exit 1; fi;

# Заново устанавливаем владельца домашней папки и data
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "chown -R postgres:postgres /home/postgres"  -noout
if [ $? = "1" ]; then exit 1; fi;
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "chown -R postgres:postgres ${db_data_path}" -noout
if [ $? = "1" ]; then exit 1; fi;

# Заново устанавливаем владельца папки ssh, для возможности запуска ssh сервера
bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "chown -R postgres:postgres /etc/ssh" -noout
if [ $? = "1" ]; then exit 1; fi;

# Установка языка по умолчанию
if ! [ "${locale}" = "ru_RU.UTF-8" ]; then
  run_cmd="sed -i 's/ru_RU.UTF-8/"${locale}"/g' /etc/default/locale"

  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}" -cn "${dks_cont_name}" -c "${run_cmd}" -noout
  if [ $? = "1" ]; then exit 1; fi;

  echo -e "${sOK}Переключение локали контейнера на \"${locale}\"."
fi;

# Если файла нет, то значит не было инита кластера... выполняем init_db
if [ ! -f "${db_data_path}/data/postgresql.conf" ]; then
  echo -e "${emptE}Кластер СУБД ${db_inst_name} не существует, создаем..."

  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}"                                                                  \
                                                   -cn "${dks_cont_name}"                                                             \
                                                   -c  "${db_bin_path}/initdb --locale=${locale} -D ${db_data_path}/data -U postgres" \
                                                   -u  "postgres"                                                                     \
                                                   -noout
  if [ $? = "1" ]; then exit 1; fi;

  if [ -f "${db_data_path}/data/pg_hba.conf" ]; then
    echo -e "${sOK}Создание кластера СУБД ${db_inst_name}."
    # Взводим переменную, что postgres и dbo нужно менять пароль, согласно переданному
    admin_change_pwd="1"
  else
    echo -e "${eRR}Создание кластера СУБД ${db_inst_name}."
    echo -e "${emptE}Возможно при выполнении скрипта были ошибки или время ожидания выполнения истекло."
    echo -e "${emptE}Операция установки прервана."
    exit 1;
  fi;


  # Инициализация БД занимает какое-то время, поэтому нужно просканировать
  # создание каталогов СУБД и только потом выполнять наши действия
  # Пока попробуем посмотреть на существование файла

else
  # Каталог с данными уже существует, используем его
  # Взводим переменную, что postgres и dbo пароль менять не нужно

  echo -e "${emptE}Каталог данных СУБД ${db_data_path}/data/ существует, монтируем..."
  admin_change_pwd="0"
fi;


# Корректировка каталога PGDATA
bash ${dks_bashfile_dir}/dks_make_pgdata.sh "${dks_init_var_file}"

# ======================  Запускаем СУБД ======================================
bash ${dks_bashfile_dir}/dks_start_pgpid.sh ${dks_cont_name} \
                                            ${db_inst_name}  \
                                            ${db_bin_path}   \
                                            ${db_data_path}  \
                                            ${db_port}
# =============================================================================


# Установка пароля postgres (соединение через сокет)
if [ "${admin_change_pwd}" = "1" ]; then
  sql_cmd=${db_bin_path}"/psql -U postgres -d postgres -p ${db_port} -h /tmp -c \\\"alter role postgres password '"${db_pwd}"'\\\""

  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}"      \
                                                   -cn "${dks_cont_name}" \
                                                   -c  "${sql_cmd}"       \
                                                   -u  "postgres"         \
                                                   -noout
  if [ $? = "1" ]; then exit 1; fi;


  echo -e "${sOK}Установка пароля пользователю postgres."
fi;

# Настройка СУБД, создание dbo и т.д. (соединение через сокет), выполняем dks_pgsqlks_init.sql
cp ${dks_inst_path}/sql/dks_pgsqlks_init.sql ${db_kslib_path_host}
sql_cmd=${db_bin_path}"/psql -U postgres -d postgres -p ${db_port} -h /tmp -f ${db_kslib_path}/dks_pgsqlks_init.sql"



bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}"      \
                                                 -cn "${dks_cont_name}" \
                                                 -c  "${sql_cmd}"       \
                                                 -u "postgres"          \
                                                 -noout
if [ $? = "1" ]; then exit 1; fi;

echo -e "${sOK}Создание\коррекция пользователя dbo."

# Настройка СУБД необходимых процедур в базе postgres (соединение через сокет), выполняем dks_dbpostgres_funcs.sql
cp ${dks_inst_path}/sql/dks_dbpostgres_funcs.sql ${db_kslib_path_host}
sql_cmd=${db_bin_path}"/psql -U postgres -d postgres -p ${db_port} -h /tmp -f ${db_kslib_path}/dks_dbpostgres_funcs.sql"

bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}"      \
                                                 -cn "${dks_cont_name}" \
                                                 -c   "${sql_cmd}"      \
                                                 -u   "postgres"        \
                                                 -noout

if [ $? = "1" ]; then exit 1; fi;

echo -e "${sOK}Создание служебных процедур в БД postgres."

# Делаем одинаковый пароль с postgres у пользователя dbo (соединение через сокет)
if [ "${admin_change_pwd}" = "1" ]; then
  sql_cmd=${db_bin_path}"/psql -U postgres -d postgres -p ${db_port} -h /tmp -c \\\"alter role dbo password '"${db_pwd}"'\\\""

  bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}"      \
                                                   -cn "${dks_cont_name}" \
                                                   -c  "${sql_cmd}"       \
                                                   -u  "postgres"         \
                                                   -noout
  if [ $? = "1" ]; then exit 1; fi;

  echo -e "${sOK}Установка пароля пользователю dbo."
fi;

# Анализируем наличие FireWall и пытаемся добавить правило для открытия порта СУБД
bash ${dks_bashfile_dir}/sys/sys_firewall.sh "${db_port}" "tcp"

# Проверка доступности сервера СУБД в сети по порту
fn_file_clr ${dks_stdout}
sql_cmd=${db_bin_path}"/psql -U postgres -d postgres -p ${db_port} -h 127.0.0.1 -c 'select 1 as pgsqlks_cmd_completed;' > ${dks_stdout_cont} 2>&1"

bash ${dks_bashfile_dir}/sys/sys_exec_cont_ex.sh -cs "${contsoft}"      \
                                                 -cn "${dks_cont_name}" \
                                                 -c  "${sql_cmd}"       \
                                                 -u "postgres"

if [ ! -z "$(grep -E "pgsqlks_cmd_completed" ${dks_stdout})" ]; then
  sql_access_net="1"
  echo -e "${sOK}Сервер СУБД ${db_inst_name} доступен в сети по порту ${db_port}."
else
  sql_access_net="0"
  echo -e "${wOK}Сервер СУБД ${db_inst_name} не доступен в сети порту ${db_port}."
fi;

fn_file_clr ${dks_stdout}

# Останавливаем контейнер
${contsoft} stop -t 1 ${dks_cont_name} >/dev/null
echo -e "${sOK}Остановка контейнера ${dks_cont_name}."


# При запуске текущего скрипта без параметра "NoServ", делаем
# окружение для запуска в systemd. Это необходимо, чтобы
# после аварийного отключения или просто перезагрузки хоста
# заново проинициализировался контейнер с запускм СУБД и 
# планировщиком

# Создание автозапуска при перезагрузке хоста
if [ -f "/lib/systemd/system/${dks_serv_name}" ]; then
  fn_file_clr /lib/systemd/system/${dks_serv_name}
fi;

# Формирование содержимого файла */service
if [ "${contsoft}" = "docker"  ]; then
  requires_service="Requires=docker.service"
  after_service="After=docker.service"
else
  requires_service=""
  after_service=""
fi;


cat > /lib/systemd/system/${dks_serv_name} << EOF
[Unit]
Description=СУБД-КС ${tag_ver}:${db_port} (${contsoft}/${typeOS})
# Запускать после старта сервисов и служб
${requires_service}
After=syslog.target
After=network.target
${after_service}

[Service]
Type=forking
ExecStart=${dks_bashfile_dir}/dks_service.sh start ${db_port}
ExecStop=${dks_bashfile_dir}/dks_service.sh stop ${db_port}
RemainAfterExit=yes

# Запрет на убийство сервиса вследствие нехватки памяти и срабатывания механизма OOM:
#-1000 полный запрет (такой у sshd стоит), -100 понизим вероятность.
OOMScoreAdjust=-1000

# Таймаут в секундах, сколько ждать system отработки старт/стоп команд.
TimeoutSec=300

[Install]
#Уровень запуска - Многопользовательский режим без графики
WantedBy=multi-user.target
EOF

systemctl daemon-reload

fn_file_echo ${dks_stdout} ${emptEsh}
systemctl enable ${dks_serv_name} >/dev/null 2>${dks_stdout}
#показываем
fn_file_echo ${dks_stdout} ${emptEsh}
fn_file_clr ${dks_stdout}

systemctl start  ${dks_serv_name}

# Проверяем что служба запустилась
if systemctl -q is-active ${dks_serv_name}; then
  echo -e "${sOK}Запуск сервиса ${dks_serv_name}."
else
  echo -e "${eRR}Запуск сервиса ${dks_serv_name}."
  echo -e "${emptE}Была предпринята неудачная попытка запуска сервиса ${dks_serv_name}."
  echo -e "${emptE}Если сервис не будет запущен, то контейнер ${dks_cont_name} не запустится после перезагрузки хоста."
  echo -e "${emptE}Обязательно свяжитесь со свои системным администратором."
fi


# Удаление ненужных файлов
rm -f ${dks_stdout}
rm -f ${db_kslib_path_host}/dks_pgsqlks_init.sql
rm -f ${db_kslib_path_host}/dks_dbpostgres_funcs.sql

# восстанавливаем значение для selinux
if [ "${se_mode}" = "Enforcing" ]; then
  # переключаем обратно
  setenforce Enforcing
fi;


echo ""
exit 0;

